// Written by IVA2K
//
// Example of HTTPServer with additional features:
// * Link status indication (LED4)
// * Local file system (create index.htm page on MBED!)
// * RPC-able class (myrpc, allows remote function call that blinks LED1 N times)
//
// Instructions:
// 1  Plug MBED into BoB2 (or other ETH connector breakout)
// 2  Plug ETH connector into your network (needs DHCP to get IP address)
// 3  Power up MBED using USB cable
// 4  Install MBED serial driver (http://mbed.org/handbook/SerialPC)
// 5  Copy compiled .bin to your MBED (make sure target device selected in the compiler is correct)
// 6  Open terminal on the mbed serial port
// 7  Push MBED reset button
// 8  Terminal will display info message with mac address, followed by IP address (if connection succeeds)
// 9  Open browser and enter the following URL, inserting your MBED's IP address:
//    http://10.0.0.321/rpc/myrpc1/blink,10 (instead of 10.0.0.321 use MBED IP address from terminal)
// 10 MBED will blink the LED 10 times
//
// Notes: there are still some bugs in HTTPServer code. 
// To help fight some of them, copy a valid favicon.ico (a 16x16 icon) file to MBED.
//

#include "mbed.h"
#include "HTTPServer.h"
#include "HTTPRPC.h"
#include "HTTPFS.h"
#include "myrpc.h"

DigitalOut led1(LED1, "led1");
DigitalOut led2(LED2, "led2");
DigitalOut led3(LED3, "led3");
DigitalOut led4(LED4, "led4");
LocalFileSystem local("local");
myrpc myrpc1(LED1, "myrpc1");

extern Ethernet eth;        // eth is defined elsewhere, avoid compiler error.
Serial pc(USBTX, USBRX);
Ticker eth_timer;

#define LED_ETH_LINK(val) do { led4=val; } while (0)
void eth_link_status() {
    static bool first = true;        // Avoid duplicate IP report on the very first pass
    static int eth_link = -1;        // Last state of eth link
    int new_link = eth.link();
    if (eth_link != new_link) {
        if (new_link) {
            // From http://mbed.org/forum/post/909/
            NetServer *net = NetServer::get();
            struct ip_addr ip = net->getIPAddr();
//            struct ip_addr gw = net->getGateway();
//            struct ip_addr nm = net->getNetmask();
//            struct ip_addr dns = net->getDNS1();
            if (!first) printf("IP: %hhu.%hhu.%hhu.%hhu\r\n", (ip.addr)&0xFF, (ip.addr>>8)&0xFF, (ip.addr>>16)&0xFF, (ip.addr>>24)&0xFF);
            first = false;
        }
        else {
            printf("IP: <link down>\r\n");
        }
        LED_ETH_LINK(new_link);
        eth_link = new_link;
    }
}

int main(void) {
    char mac[6];

    led1=1;
    led2=1;
    led3=1;
    led4=1;

    char *hostname = "mbed";
    HTTPServer http(hostname);    // Use DHCP
    http.timeout(60000);

    eth.address(mac);
    pc.printf("\r\n\r\nMBED HTTPServer \"%s\" started\r\nMAC %02X:%02X:%02X:%02X:%02X:%02X\r\n",
        hostname, mac[0], mac[1], mac[2], mac[3], mac[4], mac[5]);

    Base::add_rpc_class<AnalogIn>();
    Base::add_rpc_class<AnalogOut>();
    Base::add_rpc_class<DigitalIn>();
    Base::add_rpc_class<DigitalOut>();
    Base::add_rpc_class<PwmOut>();
    Base::add_rpc_class<Timer>();
    Base::add_rpc_class<SPI>();
    Base::add_rpc_class<BusOut>();
    Base::add_rpc_class<BusIn>();
    Base::add_rpc_class<myrpc>();
    led1=0;

    http.addHandler(new HTTPRPC());
    led2=0;

    http.addHandler(new HTTPFileSystemHandler("/", "/local/"));
    led3=0;

// FIXME: BUG:: If eth is not plugged, http.bind() hangs for a while!
    http.bind();
    led4 = 0;

    pc.printf("\r");    // Add linefeed for stupid Hyperterminal
    eth_timer.attach(&eth_link_status, 0.1);
    while(1) {
        http.poll();    // Have to call this method at least every 250ms to let the http server run.
        wait(0.100);
    }
}

//END