
/*
Copyright (c) 2010 Donatien Garnier (donatiengar [at] gmail [dot] com)
 
Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:
 
The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.
 
THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
*/

#include "NTPClient.h"

#include <stdio.h>

//#define __DEBUG
#include "dbg/dbg.h"

#define NTP_PORT 123
#define NTP_CLIENT_PORT 50420
#define NTP_REQUEST_TIMEOUT 15000
#define NTP_TIMESTAMP_DELTA 2208988800ull //Diff btw a UNIX timestamp (Starting Jan, 1st 1970) and a NTP timestamp (Starting Jan, 1st 1900)

#define htons( x ) ( (( x << 8 ) & 0xFF00) | (( x >> 8 ) & 0x00FF) )
#define ntohs( x ) (htons(x))

#define htonl( x ) ( (( x << 24 ) & 0xFF000000)  \
                   | (( x <<  8 ) & 0x00FF0000)  \
                   | (( x >>  8 ) & 0x0000FF00)  \
                   | (( x >> 24 ) & 0x000000FF)  )
#define ntohl( x ) (htonl(x))

NTPClient::NTPClient() : m_state(NTP_PING), m_pCbItem(NULL), m_pCbMeth(NULL), m_pCb(NULL),
m_watchdog(), m_timeout(0), m_closed(true), m_host(), m_pDnsReq(NULL), m_blockingResult(NTP_PROCESSING)
{
  setTimeout(NTP_REQUEST_TIMEOUT);
  DBG("New NTPClient %p\r\n",this);
}

NTPClient::~NTPClient()
{
  close();
}

//High level setup functions
NTPResult NTPClient::setTime(const Host& host) //Blocking
{
  doSetTime(host);
  return blockingProcess();
}

NTPResult NTPClient::setTime(const Host& host, void (*pMethod)(NTPResult)) //Non blocking
{
  setOnResult(pMethod);
  doSetTime(host);
  return NTP_PROCESSING;
}

#ifdef __LINKER_BUG_SOLVED__
template<class T> 
NTPResult NTPClient::setTime(const Host& host, T* pItem, void (T::*pMethod)(NTPResult)) //Non blocking
{
  setOnResult(pItem, pMethod);
  doSetTime(host);
  return NTP_PROCESSING;
}
#endif

void NTPClient::doSetTime(const Host& host)
{
  init();
  m_host = host;
  if(m_host.getIp().isNull())
  {
    //DNS query required
    m_pDnsReq = new DNSRequest();
    DBG("NTPClient : DNSRequest %p\r\n", m_pDnsReq);
    m_pDnsReq->setOnReply(this, &NTPClient::onDNSReply);
    m_pDnsReq->resolve(&m_host);
    return;
  }
  if(!m_host.getPort())
  {
    m_host.setPort(NTP_PORT);
  }
  
  m_state = NTP_PING;
  Host localhost(IpAddr(127,0,0,1), NTP_CLIENT_PORT, "localhost");
  m_pUDPSocket->bind(localhost);
  set_time( 1280000000 ); //End of July 2010... just there to limit offset range
  process();
}

void NTPClient::setOnResult( void (*pMethod)(NTPResult) )
{
  m_pCb = pMethod;
}

void NTPClient::init() //Create and setup socket if needed
{
  if(!m_closed) //Already opened
    return;
  m_state = NTP_PING;
  m_pUDPSocket = new UDPSocket;
  m_pUDPSocket->setOnEvent(this, &NTPClient::onUDPSocketEvent);
  m_closed = false;
}

void NTPClient::close()
{
  if(m_closed)
    return;
  m_closed = true; //Prevent recursive calling or calling on an object being destructed by someone else
  m_watchdog.detach();
  m_pUDPSocket->resetOnEvent();
  m_pUDPSocket->close();
  delete m_pUDPSocket;
  if( m_pDnsReq )
  {
    m_pDnsReq->close();
    delete m_pDnsReq;
    m_pDnsReq = NULL;
  }
}

#define MIN(a,b) ((a)<(b))?(a):(b)
void NTPClient::process() //Main state-machine
{
  NTPPacket pkt;
  int len;
  Host host;

  switch(m_state)
  {
  case NTP_PING:
    DBG("Ping\r\n");
    //Prepare NTP Packet:
    pkt.li = 0; //Leap Indicator : No warning
    pkt.vn = 4; //Version Number : 4
    pkt.mode = 3; //Client mode
    pkt.stratum = 0; //Not relevant here
    pkt.poll = 0; //Not significant as well
    pkt.precision = 0; //Neither this one is
    
    pkt.rootDelay = 0; //Or this one
    pkt.rootDispersion = 0; //Or that one
    pkt.refId = 0; //...
    
    pkt.refTm_s = 0;
    pkt.origTm_s = 0;
    pkt.rxTm_s = 0;
    pkt.txTm_s = htonl( NTP_TIMESTAMP_DELTA + time(NULL) ); //WARN: We are in LE format, network byte order is BE
    
    pkt.refTm_f = pkt.origTm_f = pkt.rxTm_f = pkt.txTm_f = 0;
    
    //Hex Dump:
    DBG("Dump Tx:\r\n");
    for(int i = 0; i< sizeof(NTPPacket); i++)
    {
      DBG("%02x ", *((char*)&pkt + i));
    }
    DBG("");
    
    len = m_pUDPSocket->sendto( (char*)&pkt, sizeof(NTPPacket), &m_host );
    if(len < sizeof(NTPPacket))
      { onResult(NTP_PRTCL); close(); return; }
      
    m_state = NTP_PONG; 
          
    break;
  
  case NTP_PONG:
    DBG("Pong\r\n");
    while( len = m_pUDPSocket->recvfrom( (char*)&pkt, sizeof(NTPPacket), &host ) )
    {
      if( len <= 0 )
        break;
      if( !host.getIp().isEq(m_host.getIp()) )
      //if( !ip_addr_cmp( &host.getIp().getStruct(), &m_host.getIp().getStruct() ) ) //Was working like that, trying nicer impl above
        continue; //Not our packet
      if( len > 0 )
        break;
    }
    
    if(len == 0)
      return; //Wait for the next packet
    
    if(len < 0)
      { onResult(NTP_PRTCL); close(); return; }
    
    if(len < sizeof(NTPPacket)) //TODO: Accept chunks
      { onResult(NTP_PRTCL); close(); return; }
      
    //Hex Dump:
    DBG("Dump Rx:\r\n");
    for(int i = 0; i< sizeof(NTPPacket); i++)
    {
      DBG("%02x ", *((char*)&pkt + i));
    }
    DBG("");
      
    if( pkt.stratum == 0)  //Kiss of death message : Not good !
    {
      onResult(NTP_PRTCL); close(); return;
    }
    
    //Correct Endianness
    pkt.refTm_s = ntohl( pkt.refTm_s );
    pkt.refTm_f = ntohl( pkt.refTm_f );
    pkt.origTm_s = ntohl( pkt.origTm_s );
    pkt.origTm_f = ntohl( pkt.origTm_f );
    pkt.rxTm_s = ntohl( pkt.rxTm_s );
    pkt.rxTm_f = ntohl( pkt.rxTm_f );
    pkt.txTm_s = ntohl( pkt.txTm_s );
    pkt.txTm_f = ntohl( pkt.txTm_f );
    
    //Compute offset, see RFC 4330 p.13
    uint32_t destTm_s = (NTP_TIMESTAMP_DELTA + time(NULL));
    //int32_t origTm = (int32_t) ((uint64_t) pkt.origTm - NTP_TIMESTAMP_DELTA); //Convert in local 32 bits timestamps
    //int32_t rxTm = (int32_t) ((uint64_t) pkt.rxTm - NTP_TIMESTAMP_DELTA); //Convert in local 32 bits timestamps
    //int32_t txTm = (int32_t) ((uint64_t) pkt.txTm - NTP_TIMESTAMP_DELTA); //Convert in local 32 bits timestamps
   // int64_t offset = ( ( ( pkt.rxTm_s - pkt.origTm_s ) + ( pkt.txTm_s - destTm_s ) ) << 32 + ( ( pkt.rxTm_f - pkt.origTm_f ) + ( pkt.txTm_f - 0 ) ) ) / 2;
    int64_t offset = ( (int64_t)( pkt.rxTm_s - pkt.origTm_s ) + (int64_t) ( pkt.txTm_s - destTm_s ) ) / 2; //Avoid overflow
    DBG("Sent @%d\r\n", pkt.txTm_s);
    DBG("Offset: %d\r\n", offset);
    //Set time accordingly
    set_time( time(NULL) + (offset /*>> 32*/) );
    
    onResult(NTP_OK);
    close();
      
    break;
  }
}

void NTPClient::setTimeout(int ms)
{
  m_timeout = 1000*ms;
  resetTimeout();
}

void NTPClient::resetTimeout()
{
  m_watchdog.detach();
  m_watchdog.attach_us<NTPClient>(this, &NTPClient::onTimeout, m_timeout);
}

void NTPClient::onTimeout() //Connection has timed out
{
  close();
  onResult(NTP_TIMEOUT);
}

void NTPClient::onDNSReply(DNSReply r)
{
  if(m_closed)
  {
    DBG("WARN: Discarded\r\n");
    return;
  }
  
  if( r != DNS_FOUND )
  {
    DBG("Could not resolve hostname.\r\n");
    onResult(NTP_DNS);
    close();
    return;
  }
  m_pDnsReq->close();
  delete m_pDnsReq;
  m_pDnsReq=NULL;
  doSetTime(m_host);
}
  
void NTPClient::onUDPSocketEvent(UDPSocketEvent e)
{
  switch(e)
  {
  case UDPSOCKET_READABLE: //The only event for now
    resetTimeout();
    process();
    break;
  }
}

void NTPClient::onResult(NTPResult r) //Must be called by impl when the request completes
{
  if(m_pCbItem && m_pCbMeth)
    (m_pCbItem->*m_pCbMeth)(r);
  else if(m_pCb)
    m_pCb(r);
  m_blockingResult = r; //Blocking mode
}

NTPResult NTPClient::blockingProcess() //Called in blocking mode, calls Net::poll() until return code is available
{
  m_blockingResult = NTP_PROCESSING;
  do
  {
    Net::poll();
  } while(m_blockingResult == NTP_PROCESSING);
  Net::poll(); //Necessary for cleanup
  return m_blockingResult;
}
