/*
*******************************************************************************
*                              CERES CONTROLS
*                         PANAMA, REPULIC OF PANAMA
*
*  File          : main.cpp
*  Programmer(s) : Rangel Alvarado
*  Language      : ANSI-C
*  Description   : Main file for SOMO-14D (version 1) terminal controller.
*
*  Note          : The SOMO-14D is a tiny Audio-Sound module that can 
*                  play back pre-stored audio files such as voice and music 
*                  from a micro-SD memory card.
*
*  Warning       : Tested only on a STM32F030R8 mbed board.
*                  Software must have to do a call on
*                  - MsgRxISR()
*                  - SOMO14DISR()
*
*  Usage         : 1 - Open a terminal program like CoolTerm in 9600,8N1
*                  2 - Use only the following commands
*                      List of available commands:
*                      p       play the current song
*                      h       hold the current song
*                      s       stop the current song and go to low power mode
*                      n       play the next song in the queue
*                      r       play the previous song in the queue
*                      a <num> set audio to a specific song, 511 > num > 0
*                              i.e.   num = 0010
*                      +       set volume up, max 8 levels
*                      -       set volume down, max 8 levels
*
*  ----------------------------------------------------------------------------
*  HISTORY
*   DD MM AA
*   09 03 15    Created.
*   09 03 15    Modified.
*   12 03 15    Import to mbed platform.
*   15 03 15    Added to the help list the STOP command description
*******************************************************************************
*/

/*
*******************************************************************************
*                                              INCLUDE FILES
*******************************************************************************
*/

#include "includes.h"                      /* Include all header files       */


/*
*******************************************************************************
*                              CONSTANTS AND MACROS
*******************************************************************************
*/

enum SOMO_ACTION {PLAY  = 'p', HOLD  = 'h',/* The list of available commands */
                  STOP  = 's', NEXT  = 'n', 
                  PREV  = 'r', STAU  = 'a', 
                  VOLUP = '+', VOLDN = '-'
                 };


/*
*******************************************************************************
*                                            FUNCTION PROTOTYPES 
*******************************************************************************
*/
                                 
void SOMOManageAction(INT8U nbytes);       /* Check will action be performed */
void SOMO14DISR(void);                     /* Interrupt routine of busy pin  */
void MsgRxISR(void);                       /* Interrupt routine of rx char   */

                                 
/*
*******************************************************************************
*                                               MAIN FUNCTION 
*******************************************************************************
*/                               
                                 
int main() {
  INT8U action;                             /* Variable for action command    */
    
    
    
  action   = 0;                             /* Set action initially to 0      */
  SOMO14DInit(SOMO14DISR);                  /* Set up the sound module pins   */
  MsgBufInit();                             /* Initialize the buffer          */
  MsgRxISRCfg(MsgRxISR);                    /* Attach Rx IRQ function         */
  MsgPutLine((INT8U *)"SOMO Waiting for commands\r"); /* Hello Message        */
  MsgPut('>');                              /* Waiting a command sign         */
  while(1) {                                /* Repeat forever                 */
    if (!MsgRxBufEmpty()) {                 /* If there is something Received */
      action = MsgGetChar();                /* Get the action                 */
      SOMOManageAction(action);             /* Check which action to perform  */
    }
  }
}


/*
*******************************************************************************
*                           
*                 MANAGE AN ACTION ON THE SOUND MODULE AND RETURN A MESSAGE
*
*  Description : Check before a received character which action to perform
*  Arguments   : None
*  Return        : None
*  Note(s)     : None
*******************************************************************************
*/

void SOMOManageAction(INT8U action) {
           INT8U  msg[4];                    /* Buffer the audio no.            */
           INT8U      n;                     /* Iterator                        */
    static INT16U   song;                    /* Current song to 0               */
    static INT8U vol = 7;                    /* Volume to max                   */
    

    switch (action) {                        /* Manage action                   */
      case PLAY:                             /* Play the current song           */
        SOMO14DSetAudio(song);               /* Set the current song and play it*/
        MsgPutLine((INT8U *)"PLAY\r>");      /* Execution message               */
        break;                               /* Terminate action                */
      case HOLD:                             /* Hold/Release the current song   */
        SOMO14DPause();                      /* Pause/Play the song             */
        MsgPutLine((INT8U *)"HOLD\r>");      /* Execution message               */
        break;                               /* Terminate action                */
      case STOP:                             /* Stop the song                   */
        SOMO14DStop();                       /* Quit playing the song           */
        MsgPutLine((INT8U *)"STOP\r>");      /* Execution message               */
        break;                               /* Terminate action                */
      case NEXT:                             /* Play the next song              */
        SOMO14DSetAudio(++song);             /* Increment song and play         */
        MsgPutLine((INT8U *)"NEXT SONG\r>"); /* Execution message               */
        break;                               /* Terminate action                */
      case PREV:                             /* Play the previous song          */
        SOMO14DSetAudio(--song);             /* Decrement song and play         */
        MsgPutLine((INT8U *)"PREVIOUS SONG\r>"); /* Execution message           */
        break;                               /* Terminate action                */
      case STAU:                             /* Set the audio                   */
        MsgPutLine((INT8U *)"SET AUDIO TO: ");   /* Step 1 message              */
        for(n = 0; n < 4; n++) {             /* Repeat for 4 numbers            */
          while(MsgRxBufEmpty()) {           /* Wait if is empty                */
            ;
          }
          msg[n] = MsgGetChar();             /* Hold the current no. in buffer  */
          MsgPut(msg[n]);                    /* Print the received char         */
        }
        MsgPutLine((INT8U *)".ad4\r>");      /* Print an extra string (format)  */
        song = atoi((const char *)msg);      /* Convert ASCII to number         */
        SOMO14DSetAudio(song);               /* Play the song                   */
        break;                               /* Terminate action                */
      case VOLUP:                            /* Increase volume                 */
        SOMO14DSetVol(&(++vol));             /* Alter reference, increment vol  */
        MsgPutLine((INT8U *)"VOLUME ");      /* Execution message               */
        MsgPut(vol + '0');                   /* ASCII representation of numbers */
        MsgPutLine((INT8U *)"\r>");          /* Print a command line            */
        break;                               /* Terminate action                */
      case VOLDN:                            /* Decrease volume                 */
        SOMO14DSetVol(&(--vol));             /* Alter reference, decrement vol  */
        MsgPutLine((INT8U *)"VOLUME ");      /* Execution message               */
        MsgPut(vol + '0');                   /* ASCII representation of numbers */
        MsgPutLine((INT8U *)"\r>");          /* Print a command line            */
        break;                               /* Terminate action                */
      default:                               /* Not a command? Print the help   */
        MsgPutLine((INT8U *)"NOT A COMMAND\r>");
        MsgPutLine((INT8U *)"List of available commands:\r");
        MsgPutLine((INT8U *)"p       play the current song\r");
        MsgPutLine((INT8U *)"h       hold the current song\r");
        MsgPutLine((INT8U *)"s       stop the current song\r");
        MsgPutLine((INT8U *)"n       play the next song in the queue\r");
        MsgPutLine((INT8U *)"r       play the previous song in the queue\r");
        MsgPutLine((INT8U *)"a <num> set audio to a specific song, 511 > num > 0\r");
        MsgPutLine((INT8U *)"+       set volume up, max 8 levels\r");
        MsgPutLine((INT8U *)"-       set volume down, max 8 levels\r>");
        break;                               /* Terminate action                */
  }
}


/*
*******************************************************************************
*                           
*                      RX INTERRUPT SUBROUTINE REQUEST
*
*  Description : Get the received character and insert it into the buffer
*  Arguments   : None
*  Return        : None
*  Note(s)     : None
*******************************************************************************
*/

void MsgRxISR(void) {
  MsgPutRxChar(MsgGet());                  /* Put a byte on the buffer        */
}


/*
*******************************************************************************
*                           
*                SOMO14D INTERRUPT SUBROUTINE REQUEST ATTACH
*
*  Description : Interrupt request when Busy pins goes from HIGH to LOW
*  Arguments   : None
*  Return      : None
*  Notes       : Busy pin is active low triggered.
*******************************************************************************
*/

void SOMO14DISR(void) {
  MsgPutLine((INT8U *)"Audio finish\r>"); /* Message on interrupt request     */
}
