/*
*******************************************************************************
*                              CERES CONTROLS
*                         PANAMA, REPULIC OF PANAMA
*
*  File          : SOMO14D.h
*  Programmer(s) : Rangel Alvarado
*  Language      : ANSI-C
*  Description   : SOMO14D is an acronym of SOund MOdule 1 4D SYSTEMS INC.
*                  The file manage all functions of the SOMO.
*
*  Note          : Dependancies of mbed libraries...
*                  - InterruptIn
*                  - DigitalOut
*
*  ----------------------------------------------------------------------------
*  HISTORY
*   DD MM AA
*   09 03 15    Created.
*   09 03 15    Modified.
*   12 03 15    Import to mbed platform.
*   13 03 15    Moved the SOMO Serial Out function declaration to this file.
*******************************************************************************
*/

/*
*******************************************************************************
*                              INCLUDE FILES
*******************************************************************************
*/

#include    "includes.h"                 /* Main include file                 */

InterruptIn SOMO14DBusy(SOMO_BUSY_PIN);  /* BUSY signal. Active HIGH output.  */
DigitalOut  SOMO14DData(SOMO_DATA_PIN);  /* Serial DATA input from host MCU   */
DigitalOut  SOMO14DClk(SOMO_CLK_PIN);    /* Serial Clock input from host MCU  */
DigitalOut  SOMO14DReset(SOMO_RESET_PIN);/* Master Reset. Active LOW trigger. */


/*
*******************************************************************************
*                        PRIVATE FUNCTION DECLARATION
*******************************************************************************
*/

void  SOMO14DSerialOut(INT16U SOMO14Duint16); /* Serial out command          */


/*
*******************************************************************************
*                           
*                             SOMO14D IS BUSY
*
*  Description : Check if a function is executing by the module
*  Arguments   : None
*  Return      : TRUE    if the module is holding an action
*                FALSE   if the module is in waiting state
*  Notes       : Busy pin is active low triggered.
*******************************************************************************
*/

INT8U SOMO14DisBusy(void) {
     if(SOMO14DBusy.read()) {            /* If input is HIGH                  */
         return TRUE;                    /* SOMO is doing something           */
     }
     return FALSE;                       /* SOMO is on low power mode         */
} 

/*
*******************************************************************************
*                           
*                           INITIALIZE THE SOUND MODULE
*
*  Description : Initial setup for the module and signals
*  Arguments   : pfunc    pointer to function of an ISR
*  Return      : None
*  Notes       : Activity pins are in idle state
*******************************************************************************
*/

void SOMO14DInit(void (*pfunc)(void)) {
    SOMO14DBusy.disable_irq();           /* disble busy pin interrupt        */
    SOMO14DReset = HIGH;                 /* Reset the SOMO                   */
    SOMO14DClk   = HIGH;                 /* Hold line up                     */
    SOMO14DData  = HIGH;                 /* Hold data up                     */
    SOMO14DBusy.fall(pfunc);             /* Attach an ISR                    */
    SOMO14DBusy.mode(PullUp);            /* Pull up this pin                 */
    SOMO14DBusy.enable_irq();            /* Enable the pin IRQ               */
}

/*
*******************************************************************************
*                           
*                         SOUND MODULE SERIAL OUT COMMAND
*
*  Description : Send the command, left to right, first is B15, last B0.
*  Arguments   : SOMO14Duint16    16-bit command that makes an action
*  Return      : None
*  Notes       : None
*******************************************************************************
*/

void SOMO14DSerialOut(INT16U SOMO14Duint16) {     
    INT16U bit;                          /* Handles bit variable             */
    INT8U i;                             /* Iterator                         */

    
    SOMO14DClk = LOW;                    /* CLK = 0                          */
    wait(SOMO_WAIT_BIT);                 /* Waits between bits               */
    for (i = 0; i < 16; i++) {           /* Extract the 16 bits              */
        bit = (SOMO14Duint16 & (0x8000 >> i)); /* get a bit                  */
        if (bit) {                       /* if flagged                       */
            SOMO14DData = HIGH;          /* DATA = 1                         */
        } else {                         /* Otherwise                        */
            SOMO14DData = LOW;           /* DATA = 0                         */
        } 
        wait(SOMO_WAIT_BIT);             /* Waits between bits               */
        SOMO14DClk = HIGH;               /* CLK = 1                          */
        wait(SOMO_WAIT_BIT);             /* Waits between bits               */
        SOMO14DClk = LOW;                /* CLK = 0                          */
    }
    wait(SOMO_WAIT_IDLE);                /* Idle State                       */
    SOMO14DClk = HIGH;                   /* Return clock to idle             */
}


/*
*******************************************************************************
*                           
*                         PAUSE THE SOUND MODULE
*
*  Description : Hold or release signal for the sound module (pause/play)
*  Arguments   : None
*  Return      : None
*  Notes       : None
*******************************************************************************
*/

void SOMO14DPause(void) {
  SOMO14DSerialOut(SOMO_PAUSE_CMD_OFFSET); /* Send Pause Command             */
}

/*
*******************************************************************************
*                           
*                           STOP THE SOUND MODULE
*
*  Description : Send a stop signal and goes to low consumption mode
*  Arguments   : None
*  Return      : None
*  Notes       : None
*******************************************************************************
*/

void SOMO14DStop() {
    SOMO14DSerialOut(SOMO_STOP_CMD_OFFSET); /* Send Stop Command             */
} 

/*
*******************************************************************************
*                           
*                   SET A VOLUME LEVEL ON THE SOUND MODULE
*
*  Description : Set the volume level of the SOMO.
*  Arguments   : SOMO14DVol    volume level
*  Return      : None
*  Notes       : 0 = Minimun volume
*                7 = Maximum volume
*                Alters and truncate the variable, because is pointer based.
*******************************************************************************
*/

void SOMO14DSetVol(INT8U *SOMO14DVol) {
    INT16U vol;                          /* Volume local variable            */
    
    
    if (*SOMO14DVol == SOMO_VOL_UNF) {   /* If volume is underflowed         */
      *SOMO14DVol = SOMO_VOL_MIN;        /* Set the volume to mute           */
    }
    if (*SOMO14DVol > SOMO_VOL_MAX) {    /* If volume is maximum             */
      *SOMO14DVol = SOMO_VOL_MAX;        /* Stay on maximum                  */
    }
    vol = SOMO_VOL_CMD_OFFSET + *SOMO14DVol; /* Set volume level             */
    SOMO14DSerialOut(vol);               /* Send volume command              */
}


/*
*******************************************************************************
*                           
*                     SET AN AUDIO FILE ON THE PLAYLIST
*
*  Description : Play the audio file that you want to play if exists
*  Arguments   : SOMO14DAudioFileNo    audio file to play
*  Return      : None
*  Notes       : 0 = Min, 511 = Max.  Total = 512 files.
*******************************************************************************
*/

void SOMO14DSetAudio(INT16U SOMO14DAudioFileNo) {
    if (SOMO14DAudioFileNo > SOMO_SONG_CMD_MAX) {/* If file is over max      */
        SOMO14DAudioFileNo = SOMO_SONG_CMD_MAX;  /* Trunk to max             */
    } 
    SOMO14DSerialOut(SOMO14DAudioFileNo);        /* Fetch the file and play  */
}
