//
// User interface implementation
//

#ifndef _USERINTERFACE_
#define _USERINTERFACE_

#ifndef _UIMENU_
#include "UIMenu.h"
#endif

#ifndef _SETTINGSMENU_
#include "SettingsMenu.h"
#endif

class HomeMenu;
class LightString;

// The ControllerUI is a pure virtual class describing the
// functions one of the light selectors needs to provide
class ControllerUI
{
public:
    ControllerUI( CheapLCD * lcd, LightString * lights )
    : fLCD( lcd ), fLights( lights ), fLightsOn( true ) {};
    
    virtual ~ControllerUI() {}
    
    virtual void Display( bool on ) = 0;
    virtual int32_t KnobMoved( int32_t step ) = 0;

    virtual void LightSwitch( bool on );
    virtual void TurnOn() = 0;
    virtual bool AreLightsOn() const { return fLightsOn; }

protected:
    CheapLCD * fLCD;
    LightString * fLights;
    bool fLightsOn;
};

// PatternSelector lets you choose patterns stored on the SD card
typedef vector<uint32_t> Pattern;

class PatternSelector : public ControllerUI
{
public:
    PatternSelector( CheapLCD * lcd, LightString * lights );

    virtual ~PatternSelector() {};
    
    virtual void Display( bool on );
    virtual int32_t KnobMoved( int32_t step );
    
    virtual void TurnOn();

private:
    void DrawPattern();
    void SetLights();
    
    vector<string> fPatternNames;
    vector<Pattern> fPatterns;
};
    
// This selector lets you choose among saturated colors
class ColorSelector : public ControllerUI
{
public:
    ColorSelector( CheapLCD * lcd, LightString * lights ) 
    : ControllerUI( lcd, lights )
    {}
    
    virtual ~ColorSelector() {};
    
    virtual void Display( bool on );
    virtual int32_t KnobMoved( int32_t step );
    
    virtual void TurnOn();

private:
    void DrawCursor( bool drawSprite );
};

// This selector lets you choose white lights
// with a variety of brightness levels
class WhiteSelector : public ControllerUI
{
public:
    WhiteSelector( CheapLCD * lcd, LightString * lights ) 
    : ControllerUI( lcd, lights )
    {}
    
    virtual ~WhiteSelector() {};
    
    virtual void Display( bool on );
    virtual int32_t KnobMoved( int32_t step );
    
    virtual void TurnOn();

private:
    void SetLights();
    void DrawLevel();
};

// This is the generic interface the UI uses to choose a
// selector and talk to it.
typedef enum { kColorSelector = 0, kWhiteSelector = 1, kPatternSelector = 2 } ESelector;

class LightController : public PushKnobUI
{
public:
    LightController( HomeMenu * parent, CheapLCD * lcd, LightString * lights );
    
    virtual void Display( bool on ) { fSubController->Display( on ); }
    void LightSwitch( bool on ) { fSubController->LightSwitch( on ); }
    bool AreLightsOn() const { return fSubController->AreLightsOn(); }
    
    void SetSelector( ESelector sel );
  
protected:

    virtual int32_t KnobMoved( int32_t step );
    virtual void KnobPushed();
    
    SETUP_KNOB_CALLBACKS( LightController )

private:
    HomeMenu * fParent;
    LightString * fLights;
    
    ControllerUI * fSubController;
    ColorSelector fColorSelector;
    WhiteSelector fWhiteSelector;
    PatternSelector fPatternSelector;
};

// This implements the home-page menu
//
class HomeMenu : public UIMenu
{
public:
    HomeMenu( CheapLCD * lcd, LightString * lights );
    virtual ~HomeMenu() {};
    
    void SetLightsOn( bool isOn );
    
protected:
    void DoLightController( ESelector sel );
    void DoToggleLightsOn();

    virtual void KnobPushed();    
    virtual void AttachButton( PushButton * button ) { button->attach( this, &HomeMenu::KnobPushed ); }
    
    LightController fLightController;
    SettingsMenu fSettingsMenu;

private:
    bool fForcedOff;
};
    
#endif

