//
// Click-knob menu implementation
//

#ifndef _UIMENU_
#define _UIMENU_

#include <vector>
#include <string>

#ifndef _ROTARY_ENCODER_
#include "RotaryEncoder.h"
#endif

#ifndef _PUSHBUTTON_
#include "PushButton.h"
#endif

class CheapLCD;

#define SETUP_KNOB_CALLBACKS( className ) \
    virtual void AttachKnob( RotaryEncoder * knob ) { knob->attach( this, &className::KnobMoved ); }\
    virtual void AttachButton( PushButton * button ) { button->attach( this, &className::KnobPushed ); }
    
#define SETUP_KNOBPUSH_CALLBACK( className ) \
    virtual void AttachButton( PushButton * button ) { button->attach( this, &className::KnobPushed ); }

// This class manages the input hardware
class PushKnobUI
{
public:
    PushKnobUI( CheapLCD * lcd ) :  fLCD( lcd ), fKnob( NULL ), fKnobButton( NULL ) {}
    virtual ~PushKnobUI() {}

    // Call this with pointers to the devices when the menu is active.
    // When another screen is active, pass NULL pointers to de-activate
    // the menu.
    virtual void ConnectDevices( RotaryEncoder * knob, PushButton * button );
    
    // Hands over the control from one control item to another
    // (e.g., from a menu to a submenu).  Control actions are routed
    // to the other item)
    virtual void SwitchControl( PushKnobUI * otherDevice );
    
    // This gets called any time there's control
    // activity, it wakes up the LCD backlight.
    virtual void Wake();
    
    virtual bool IsSleeping() const;

protected:

    virtual int32_t KnobMoved( int32_t step );
    virtual void KnobPushed();
    virtual void Sleep();
    
    SETUP_KNOB_CALLBACKS( PushKnobUI )

    CheapLCD * fLCD;
     
private:
    RotaryEncoder * fKnob;
    PushButton * fKnobButton;
};

// This knob implements a basic menu system
class UIMenu : public PushKnobUI
{
public:
    static const int kNoSelection = -1;

    UIMenu( CheapLCD * lcd, const char * header = NULL, bool upMenuItem = false );
    virtual ~UIMenu() {}
    
    // Call to add an item to the menu
    virtual void AddItem( const char * label );
    
    // Turn on / check status the display of the menu
    virtual void Display( bool on );
    virtual bool IsDisplayOn() const { return fDisplayOn; }
    
    // Switch to another menu
    virtual void SwitchTo( UIMenu * nextMenu );
    
    // Returns the currently selected item
    virtual int SelectedItem() { return fSelectedItem; }
    
    // Force select item (simulates selecting and click it w/ the knob)
    virtual void SelectItem( int i ) { fSelectedItem = i; Display( true ); }
    
    // Change the text of an item (updates display if display is on)
    virtual void ChangeItem( int item, const char * label );
    
    // Change the header (title) of the menu
    virtual void ChangeHeader( const char * header );
    
protected:
    vector<string> fLabels;
    string fHeader;
    
    virtual int32_t KnobMoved( int32_t step );
    virtual void KnobPushed();
    virtual void ClearItem( int i, uint32_t color );

    SETUP_KNOB_CALLBACKS( UIMenu )

private:
    void DrawItem( int item );
    void DrawHeader();
    
    bool fDisplayOn;
    int fSelectedItem;
};

#endif
