/*
  NokiaLCD.h - Library for a Nokia LCD with the epson driver.
  Created by Thomas Jespersen, July 2009 (Originally Arduino Sketch by Gravitech.us)
  Released into the public domain.
*/
#ifndef NokiaLCD_h
#define NokiaLCD_h

#ifndef MBED_H
#include "mbed.h"
#endif

#ifndef __HOLDINTERRUPTS__
#include "HoldInterrupts.h"
#endif

typedef unsigned char byte;

// Full color image
typedef struct image_data_t 
{
    const unsigned char *pixel_data;
    uint32_t numRows, numCols;
} image_data_t;

// A "sprite" just defines a mask for combining two colors
typedef struct sprite_data_t 
{
    const unsigned char * pixel_data;
    uint32_t numRows, numCols;
} sprite_data_t;

// Glyphs are like sprites, with text baseline information.
typedef struct glyph_data_t
{
    const unsigned char * pixel_data;
    uint32_t numRows, numCols;
    int32_t baseline;
} glyph_data_t;

// This class provides a way to fade in/out the backlight 

class LCDFadeOut
{
public:
    LCDFadeOut( PinName pwmPin ) : fPWM( pwmPin )
    {
        fPWM.period( 0.002 );   // Fairly fast (500 cycles/sec)
        fPWM = 0.0;             // default on
        fState = true;
    }
    
    ~LCDFadeOut() { }
    
    void Fade( bool on )
    {
        if (on == fState)   // no change
            return;

        fState = on;
        HoldInterrupts noirq();
        
        const int numSteps = 40;
        const float maxStep = (numSteps - 1.0);

        // For some reason, the fade looks better with asymetric times.
        int delay = (on ? 300 : 400) / numSteps;  // Total time to fade, in ms, divided by steps
        
        for (int i = 0; i < numSteps; ++i)
        {
            wait_ms( delay );
            // Note the line is pulled -low- to turn the display on
            fPWM = (!on) ? i / maxStep : (maxStep - i) / maxStep;
        }
    }
    
    // Switch w/o fading
    void Switch( bool on )
    {
        if (on == fState)
            return;
            
        fState = on;
        fPWM = (!fState) ? 1.0 : 0.0;
    }
    
    bool BacklightOn() const { return fState; }        

private:
    bool fState;
    PwmOut fPWM;
};

class CheapLCD
{
 public:
    CheapLCD( PinName mosi, PinName sclk, 
              PinName cs, PinName rst );

    void put_pixel(byte color, byte x, byte y);
    void set_box(byte x1, byte y1, byte x2, byte y2);
    void clear(uint32_t color, byte x1, byte y1, byte x2, byte y2);
    void draw_color_bar(void);
    void erase(void);
    
    // This draws small, aliased, 8x8 characters
    void draw_text_line(uint32_t fcolor,
                        uint32_t bcolor,
                        byte x, byte y,char c);
    void draw_text(uint32_t fcolor,
                   uint32_t bcolor,
                   byte x, byte y,char *text);
                   
    // This draws large, high quality anti-aliased text.
    // Note "y" refers to the text baseline
    void draw_glyph_text( uint32_t fcolor, uint32_t bcolor,
                          int x, int y, const char * str );
                          
    // Draw anti-aliased numbers (smaller than the
    // general text)
    void draw_number( uint32_t fcolor,
                      uint32_t bcolor, byte x, byte y,
                      int number );
    
    // Copy image data to the screen
    void copy_screen( const image_data_t * screenData );

    // Draw a "splash" image, covering the whole screen
    // (screen data is hard coded, the argument is an index to them)
    void splash(int demo);
    
    // Backlight control
    void fade_backlight( bool on )  { fBacklight.Fade( on ); }
    void switch_backlight( bool on ){ fBacklight.Switch( on ); }
    bool is_backlight_on() const    { return fBacklight.BacklightOn(); }

    // Guts of the sprite/text rendering
    void draw_sprite( uint32_t fcolor, uint32_t bcolor,
                      byte x, byte y, 
                      const sprite_data_t * sprite );
    void draw_glyph( uint32_t fcolor, uint32_t bcolor,
                      byte x, byte y, 
                      const glyph_data_t * sprite );
    void send_packed_pixels( uint32_t pixel_data[], uint32_t numPixels );
    
    // TEST CODE
    void demo_number( int number );
    void ramp( uint32_t color, uint32_t y, uint32_t height );

    friend class ChipSelect;

 private:

    void sendData( byte data );
    void sendCMD( byte data );
    void init();
    
    void draw_mask( uint32_t fcolor, uint32_t bcolor,
                      byte x, byte y, 
                      uint32_t numRows, uint32_t numCols,
                      const unsigned char * pixel_data );

    SPI fSPI;
    DigitalOut fReset;
    DigitalOut fCS;
    int fCSLevel;
    
    LCDFadeOut fBacklight;
};

//*******************************************************
//                12-Bit Color Definitions
//*******************************************************
#define WHITE    0xFFF
#define BLACK    0x000
#define RED        0xF00
#define GREEN    0x0F0
#define BLUE    0x00F
#define CYAN    0x0FF
#define MAGENTA    0xF0F
#define YELLOW    0xFF0
#define BROWN    0xB22
#define ORANGE    0xFA0
#define PINK    0xF6A
#define GRAY    0x888

const int kColorWheelImg = 1;

#endif
