//
// Tools for talking to the CuriLight string
//

#ifndef __LIGHTSTRING__
#define __LIGHTSTRING__

#ifndef MBED_H
#include "mbed.h"
#endif

#ifndef __LIGHTSNOOP__
#include "LightSnoop.h"
#endif

#include <vector>

class LightString
{
public:

    // Note: We rely on the default serial configuration of 9600 8N1,
    // this matches the CuriLights serial parameters.
    LightString( PinName pin, int numLights );

    // This sends an initialization to the string and configures it.
    // If numLights is non-zero, the length of the light string is reset
    void InitLights( int numLights = 0 );
    
    int GetNumLights() const { return fNumLights; }
    
    // The light code uses a "decimal RGB" notation, where
    // the color is specified as a three digit number, with each
    // digit from 0..7.  The first (hundreds) digit specifies
    // red, the second (tens) digit specifies green, and
    // the last (ones) blue.  So, for example, "700" specifies
    // red, and "755" is pink.
    
    void SetAllLights( int color );
    
    // Set a single light (id == the Nth light in the string, starting from zero)
    void SetOneColor( int color, uint8_t id );
    
    // Set each light to the value in the corresponding array.
    void SetColors( const vector<int>& colorList );
    
    void Off()      { SetAllLights( 0 ); }
    
    void Red()      { SetAllLights( 700 ); }
    
    void Green()    { SetAllLights( 70 ); }
    
    void Blue()     { SetAllLights( 7 ); }
    
    void Ouch()     { SetAllLights( 777 ); }
    
    LightSnoop * Snoop() { return &fSnoop; }
    
    void FlushOutput() { HandleOutgoingData(); };
    
    void Debug();
    
private:
    void sendCommand1( uint8_t ch );
    void sendCommand2( uint8_t ch1, uint8_t ch2 );
    void sendCommand3( uint8_t ch1, uint8_t ch2, uint8_t ch3 );
    
    void HandleIncomingData();
    void HandleOutgoingData();
    
    // Note the top red bit is stripped off to fit in the byte, other
    // code must manually add it back to the top bit if the light ID
    uint8_t colorByte( uint8_t r, uint8_t g, uint8_t b )
    {   return ((r & 3) << 6) | (g << 3) | b;  }
    
    uint8_t colorByte( int rgb, uint8_t& redBit )
    {   redBit = ((rgb / 100) >> 2) << 7;
        return colorByte( rgb/100, (rgb % 100) / 10, rgb % 10 );
    }
    
/*    uint8_t color( unsigned char * rgb, uint8_t& redBit )
    {
        redBit = ((rgb[0] - '0') >> 2) << 7;
        return color( rgb[0] - '0', rgb[1] - '0', rgb[2] - '0' );
    }
*/    
    Serial fLightsPort;
    Serial fUSBPort;
    int fNumLights;
    LightSnoop fSnoop;
    uint8_t fBuffer[384];
    int fBufferInPos;
    int fBufferOutPos;
};
#endif
