/*
The MIT License (MIT)

Copyright (c) 2016 British Broadcasting Corporation.
This software is provided by Lancaster University by arrangement with the BBC.

Permission is hereby granted, free of charge, to any person obtaining a
copy of this software and associated documentation files (the "Software"),
to deal in the Software without restriction, including without limitation
the rights to use, copy, modify, merge, publish, distribute, sublicense,
and/or sell copies of the Software, and to permit persons to whom the
Software is furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
DEALINGS IN THE SOFTWARE.
*/

#ifndef MICROBIT_ULTRASOUND_H
#define MICROBIT_ULTRASOUND_H

#include "mbed.h"
#include "MicroBitComponent.h"
#include "TimedInterruptIn.h"
// I need to include MicroBitPin.h exclusively to import all pin names.
#include "MicroBitPin.h"

/*
 * Message Bus ID, assuming this value is not taken
 */
#define MICROBIT_ID_ULTRASOUND                            50

/*
 * Configuration constants
 */
#define MICROBIT_ULTRASOUND_TRIGGER_DURATION_DEFAULT_US   12
#define MICROBIT_ULTRASOUND_IDLE_DURATION_DEFAULT_US      2
#define MICROBIT_ULTRASOUND_PERIOD_DEFAULT_MS             600
#define TRIGGER_ACTIVE_VALUE                              1
#define MICROBIT_ULTRASOUND_PULLMODE_DEFAULT              PullUp

/*
 * Ultrasound events
 */
#define MICROBIT_ULTRASOUND_EVT_TRIGGER                       1
#define MICROBIT_ULTRASOUND_EVT_ECHO_RISE                     2
#define MICROBIT_ULTRASOUND_EVT_ECHO_FALL                     3
#define MICROBIT_ULTRASOUND_EVT_ECHO_EDGE                     4
#define MICROBIT_ULTRASOUND_EVT_ECHO_PULSE_HI                 5
#define MICROBIT_ULTRASOUND_EVT_ECHO_PULSE_LO                 6
#define MICROBIT_ULTRASOUND_EVT_TEST                          7

/*
 * Component's status flags
 */
#define ULTRASOUND_STATUS_ADDED_TO_IDLE                0x02
#define ULTRASOUND_STATUS_EVENTON_RISE                 0x04
#define ULTRASOUND_STATUS_EVENTON_FALL                 0x08
#define ULTRASOUND_STATUS_EVENTON_PULSE                0x10
#define ULTRASOUND_STATUS_EVENTON_TRIGGER              0x20
#define ULTRASOUND_STATUS_EVENTON_EDGE                 0x40


/**
  * Class definition for MicroBit Ultrasonic distance sensor
  *
  * Infers and stores the distance reported by ultrasonic sensor attached
  * to the micro:bit. The library has been tested with sensor HC-SR04
  *
  */
class MicrobitUltrasound : public MicroBitComponent
{
  unsigned long           sampleTime;
  uint16_t                triggerPeriod_ms;
  uint16_t                triggerDuration_us;
  bool                    triggerActiveValue;

  //Pins
  DigitalOut              trigger;
  TimedInterruptIn        echo;

  public: 

  /**
   * Constructor.
   * Create new MicrobitUltrasound that reports distance to closest object
   *
   * @param triggerPinName Name of the MicroBitPin to be used as trigger.
   *
   * @param echoPinName Name of the MicroBitPin to be used as response from sensor.
   *
   * @code
   * MicrobitUltrasound uSoundSensor(MICROBIT_PIN_P1, 100, true, MICROBIT_PIN_P2, PullUp);
   * @endcode
   */
  MicrobitUltrasound(PinName triggerPinName, int _triggerPeriod_ms, bool _triggerActiveValue,
      PinName echoPinName, PinMode echoPinMode, uint16_t id = MICROBIT_ID_ULTRASOUND);

  /**
   * Constructor.
   * Create new MicrobitUltrasound that reports distance to closest object
   *
   * @param triggerPinName Name of the MicroBitPin to be used as trigger.
   *
   * @param echoPinName Name of the MicroBitPin to be used as response from sensor.
   *
   * @code
   * MicrobitUltrasound uSoundSensor(MICROBIT_PIN_P1, MICROBIT_PIN_P2);
   * @endcode
   */
  MicrobitUltrasound(PinName triggerPinName, PinName echoPinName, uint16_t id = MICROBIT_ID_ULTRASOUND);

  /**
   * Destructor for MicrobitUltrasound, where we deregister this instance from the array of fiber components.
   */
  ~MicrobitUltrasound();

  /**
   * Set Echo PinMode
   */
  void setEchoPinMode(PinMode echoPinMode);

  /**
   * Set the sample rate at which the micro:bit triggers sensor (in ms).
   *
   * The default sample period is 1 second.
   *
   * @param period the requested time between triggers, in milliseconds.
   *
   * @note the micro:bit sends triggers in the background.
   */
  void setPeriod(int period_ms);

  /**
   * Get the sample rate at which the micro:bit triggers sensor (in ms).
   *
   * @return period the requested time between triggers, in milliseconds.
   */
  int getPeriod();


  /**
   * Start periodic trigger to sensor. This call also will add the class
   * to fiber components for periodic callbacks.
   *
   * @return MICROBIT_OK on success.
   */
  int start(void);

  /**
   * Stop periodic trigger to sensor. 
   *
   * @return MICROBIT_OK on success.
   */
  int stop(void);

  /**
   * Enables a particular event for the MicrobitUltrasound
   *
   * @return 1 if we're due to take a new trigger, 0 otherwise.
   */
  int eventOn(int eventType);

  /**
   * Send trigger to start measurement in sensor
   *
   */
  void fireTrigger(void);

  /**
   * Periodic callback from MicroBit idle thread.
   */
  virtual void idleTick();

  private:

  /**
   * Determines if we're due to take another reading
   *
   * @return 1 if we're due to take a trigger, 0 otherwise.
   */
  int isTriggerNeeded();

  /**
   * Interrupt handler for when an rise interrupt is triggered.
   */
  void onEchoRise();

  /**
   * Interrupt handler for when an fall interrupt is triggered.
   */
  void onEchoFall();

  /**
   * This member function manages the calculation of the timestamp of a pulse detected
   * on the echo pin.
   *
   * @param eventValue the event value to distribute onto the message bus.
   */
  void pulseWidthEvent(int eventValue);


  /**
   * Update Sensor measurement
   *
   * @return MICROBIT_OK on success.
   */
  int updateSample(void);
};

#endif
