#ifndef _GSM_H_
#define _GSM_H_

#include <stdio.h>
#include "mbed.h"

#define DEFAULT_TIMEOUT         5
#define SMS_MAX_LENGTH          16


enum GSM_MESSAGE {
    MESSAGE_RING = 0,
    MESSAGE_SMS  = 1,
    MESSAGE_ERROR
};


/** GSM class.
 *  Used for mobile communication. attention that GSM module communicate with MCU in serial protocol
 */
class GSM
{
public:
    /** Create GSM instance
     *  @param tx  uart transmit pin to communicate with GSM module
     *  @param rx  uart receive pin to communicate with GSM module
     *  @param baudRate baud rate of uart communication
     *  @param number default phone number during mobile communication
     */
    GSM(PinName tx, PinName rx, int baudRate,char *number) : gprsSerial(tx, rx) {
        //gprsSerial.baud(baudRate);
        phoneNumber = number;
    };
    
    int powerCheck(void);
    
    /** init GSM module including SIM card check & signal strength & network check
     *  @returns
     *      0 on success,
     *      -1 on error
     */
    int init(void);
    
    /** Register Network of GSM module including signal strength & network check
     *  @returns
     *      0 on success,
     *      -1 on error
     */
    int registerNet(void);
    
    /** Check network status of GSM module
     *  @returns
     *      0 on success,
     *      -1 on error
     */
    int checkNetStatus(void);

    /** Check SIM card' Status
     *  @returns
     *      0 on success,
     *      -1 on error
     */
    int checkSIMStatus(void);

    /** Check signal strength
     *  @returns
     *      signal strength in number(ex 3,4,5,6,7,8...) on success,
     *      -1 on error
     */
    int checkSignalStrength(void);

    /** Set SMS format and processing mode
     *  @returns
     *      0 on success,
     *      -1 on error
     */
    int settingSMS(void);

    /** Send text SMS
     *  @param  *number    phone number which SMS will be send to
     *  @param  *data   message that will be send to
     *  @returns
     *      0 on success,
     *      -1 on error
     */
    int sendSMS(char *number, char *data);

    /** Read SMS by index
     *  @param  *message   buffer used to get SMS message
     *  @param  index    which SMS message to read
     *  @returns
     *      0 on success,
     *      -1 on error
     */
    int readSMS(char *message, int index);

    /** Delete SMS message on SIM card
     *  @param  *index    the index number which SMS message will be delete
     *  @returns
     *      0 on success,
     *      -1 on error
     */
    int deleteSMS(int index);

    /** Read SMS when coming a message,it will be store in messageBuffer.
     *  @param message  buffer used to get SMS message
     */
    int getSMS(char* message);

    /** Call someone
     *  @param  *number    the phone number which you want to call
     *  @returns
     *      0 on success,
     *      -1 on error
     */
    int callUp(char *number);

    /** Auto answer if coming a call
     *  @returns
     *      0 on success,
     *      -1 on error
     */
    int answer(void);

    /** Join GSM network
     *  @param *apn Access  Point Name to connect network
     *  @param *userName    general is empty
     *  @param *passWord    general is empty
     */
    int join(char* apn, char* userName = NULL, char* passWord = NULL);
    
    /** Disconnect from network
     *  @returns
     *      0 on success,
     *      -1 on error
     */
    int disconnect(void);
    
    /** Set blocking of the connection
     *  @param  netopen_to   time out of open the socket network in second
     *  @param  cipopen_to   time out of open the connection to server in second
     *  @param  cipsend_to   time out of send data to server in second
     *  @returns
     *      0 on success,
     *      -1 on error
     */
    int SetBlocking(int netopen_to=5, int cipopen_to=5, int cipsend_to=5);
    
    /** Build TCP connect
     *  @param  *ip    ip address which will connect to
     *  @param  *port   TCP server' port number
     *  @returns
     *      0 on success,
     *      -1 on error
     */
    int connectTCP(char *ip, int port);

    /** Send data to TCP server
     *  @param  *data    data that will be send to TCP server
     *  @returns
     *      0 on success,
     *      -1 on error
     */
    int sendTCPData(char *data, int len);
    
    /** Send data to TCP server
     *  @param  *buff    data that will be received from TCP server
     *  @param  len     size of buffer to read
     *  @returns
     *      0 on success,
     *      -1 on error
     */
    int receivedTCPData(char *buff, int len);

    /** Close TCP connection
     *  @returns
     *      0 on success,
     *      -1 on error
     */
    int closeTCP(void);
    
    /** Clear serial pipe
     */
    void purge(void);

    Serial gprsSerial;
    //USBSerial pc;

private:
    /** Read from GSM module and save to buffer array
     *  @param  *buffer buffer array to save what read from GSM module
     *  @param  *count  the maximal bytes number read from GSM module
     *  @returns
     *      0 on success,
     *      -1 on error
     */
    int readBuffer(char *buffer,int count);

    /** Send AT command to GSM module
     *  @param  *cmd command array which will be send to GSM module
     */
    void sendCmd(char *cmd);

    /** Check GSM module response before timeout
     *  @param  *resp   correct response which GSM module will return
     *  @param  *timeout    waiting seconds till timeout
     *  @returns
     *      0 on success,
     *      -1 on error
     */
    int waitForResp(char *resp, int timeout);

    /** Send AT command to GSM module and wait for correct response
     *  @param  *cmd    AT command which will be send to GSM module
     *  @param  *resp   correct response which GSM module will return
     *  @param  *timeout    waiting seconds till timeout
     *  @returns
     *      0 on success,
     *      -1 on error
     */
    int sendCmdAndWaitForResp(char *cmd, char *resp, int timeout);

    Timer timeCnt;
    char *phoneNumber;
    char messageBuffer[SMS_MAX_LENGTH];
};

#endif // _GSM_H_

