#include "SIM5360.h"

int GSM::readBuffer(char *buffer,int count)
{
    int i = 0;
    timeCnt.start();  // start timer
    while(1) {
        while (gprsSerial.readable()) {
            char c = gprsSerial.getc();
            if (c == '\r' || c == '\n') c = '$';
            buffer[i++] = c;
            if(i > count)break;
        }
        if(i > count)break;
        if(timeCnt.read() > DEFAULT_TIMEOUT) {
            timeCnt.stop();
            timeCnt.reset();
            break;
        }
    }
    wait(0.5);
    while(gprsSerial.readable()) {  // display the other thing..
        char c = gprsSerial.getc();
    }
    return 0;
}

void cleanBuffer(char *buffer, int count)
{
    for(int i=0; i < count; i++) {
        buffer[i] = '\0';
    }
}

void GSM::purge(void)
{
    while(gprsSerial.readable())
    {
        gprsSerial.getc();
    }
}

void GSM::sendCmd(char *cmd)
{
    gprsSerial.puts(cmd);
}

int GSM::waitForResp(char *resp, int timeout)
{
    int len = strlen(resp);
    int sum=0;
    timeCnt.start();

    while(1) {
        if(gprsSerial.readable()) {
            char c = gprsSerial.getc();
            sum = (c==resp[sum]) ? sum+1 : 0;
            if(sum == len)break;
        }
        if(timeCnt.read() > timeout) {  // time out
            timeCnt.stop();
            timeCnt.reset();
            return -1;
        }
    }
    timeCnt.stop();                 // stop timer
    timeCnt.reset();                    // clear timer
    while(gprsSerial.readable()) {      // display the other thing..
        char c = gprsSerial.getc();
    }

    return 0;
}

int GSM::sendCmdAndWaitForResp(char *cmd, char *resp, int timeout)
{
    sendCmd(cmd);
    return waitForResp(resp,timeout);
}

int GSM::powerCheck(void)
{
    return sendCmdAndWaitForResp("AT\r\n", "OK", 2);    
}

int GSM::init(void)
{
    int i=10;

    while (i--) {
        // purge any messages 
        purge();
        
        // check interface
        int r = sendCmdAndWaitForResp("AT\r\n", "OK", DEFAULT_TIMEOUT);
        wait_ms(100);
        if(r == 0) break;
    }
    if (i < 0) {
        goto failure;
    }

    if(0 != sendCmdAndWaitForResp("ATE0\r\n", "OK", DEFAULT_TIMEOUT))
        goto failure;

    if(0 != sendCmdAndWaitForResp("AT+CMEE=2\r\n", "OK", DEFAULT_TIMEOUT))
        goto failure;

    if(0 != checkSIMStatus()) {
        goto failure;
    }

    if(0 != sendCmdAndWaitForResp("AT+CGREG=1\r\n", "OK", DEFAULT_TIMEOUT)){
        goto failure;
    }

    if(0 != sendCmdAndWaitForResp("AT+CREG=1\r\n", "OK", DEFAULT_TIMEOUT)){
        goto failure;
    }

    if(0 != settingSMS()){
        goto failure;
    }

    return 0;

    failure:
    return -1;
}

int GSM::registerNet(void)
{
    int ret = -1;
    int i = 10;
    while(i--){
        int r = checkNetStatus();
        if(r == 0){
            ret = 0;
            break;
        }
        wait_ms(1000);
    }

    if(checkSignalStrength()<1) {
        goto failure;
    }

    failure:
    return ret;
}

int GSM::checkNetStatus(void)
{
    int netReg = -1;
    int gprsReg = -1;
    char gprsBuffer[30];
    cleanBuffer(gprsBuffer,30);

    sendCmd("AT+CREG?\r\n");
    readBuffer(gprsBuffer,30);
    if((NULL != strstr(gprsBuffer,"+CREG: 0,1"))) {
        netReg = 0;
    }

    wait_ms(100);
    sendCmd("AT+CREG?\r\n");
    readBuffer(gprsBuffer,30);
    if((NULL != strstr(gprsBuffer,"+CGREG: 0,1"))) {
        gprsReg = 0;
    }

    return netReg | gprsReg;
}

int GSM::checkSIMStatus(void)
{
    char gprsBuffer[30];
    int count = 0;
    cleanBuffer(gprsBuffer,30);
    while(count < 3) {
        sendCmd("AT+CPIN?\r\n");
        readBuffer(gprsBuffer,30);
        if((NULL != strstr(gprsBuffer,"+CPIN: READY"))) {
            break;
        }
        count++;
        wait(1);
    }

    if(count == 3) {
        return -1;
    }
    return 0;
}

int GSM::checkSignalStrength(void)
{
    char gprsBuffer[100];
    int index,count = 0;
    cleanBuffer(gprsBuffer,100);
    while(count < 3) {
        sendCmd("AT+CSQ\r\n");
        readBuffer(gprsBuffer,25);
        if(sscanf(gprsBuffer, "AT+CSQ$$$$+CSQ: %d", &index)>0) {
            break;
        }
        count++;
        wait(1);
    }
    if(count == 3) {
        return -1;
    }
    return index;
}

int GSM::settingSMS(void)
{
    if(0 != sendCmdAndWaitForResp("AT+CNMI=2,2\r\n", "OK", DEFAULT_TIMEOUT)) {
        return -1;
    }
    if(0 != sendCmdAndWaitForResp("AT+CMGF=1\r\n", "OK", DEFAULT_TIMEOUT)) {
        return -1;
    }
    return 0;
}

int GSM::sendSMS(char *number, char *data)
{
    char cmd[64];
    while(gprsSerial.readable()) {
        char c = gprsSerial.getc();
    }
    snprintf(cmd, sizeof(cmd),"AT+CMGS=\"%s\"\r\n",number);
    if(0 != sendCmdAndWaitForResp(cmd,">",DEFAULT_TIMEOUT)) {
        return -1;
    }
    wait(1);
    gprsSerial.puts(data);
    gprsSerial.putc((char)0x1a);
    return 0;
}

int GSM::readSMS(char *message, int index)
{
    int i = 0;
    char gprsBuffer[100];
    char *p,*s;
    gprsSerial.printf("AT+CMGR=%d\r\n",index);
    cleanBuffer(gprsBuffer,100);
    readBuffer(gprsBuffer,100);
    if(NULL == ( s = strstr(gprsBuffer,"+CMGR"))) {
        return -1;
    }
    if(NULL != ( s = strstr(gprsBuffer,"+32"))) {
        p = s + 6;
        while((*p != '$')&&(i < SMS_MAX_LENGTH-1)) {
            message[i++] = *(p++);
        }
        message[i] = '\0';
    }
    return 0;
}

int GSM::deleteSMS(int index)
{
    char cmd[32];
    snprintf(cmd,sizeof(cmd),"AT+CMGD=%d\r\n",index);
    sendCmd(cmd);
    return 0;
}

int GSM::getSMS(char* message)
{
    if(NULL != messageBuffer) {
        strncpy(message,messageBuffer,SMS_MAX_LENGTH);
    }
    return 0;
}

int GSM::callUp(char *number)
{
    if(0 != sendCmdAndWaitForResp("AT+COLP=1\r\n","OK",5)) {
        return -1;
    }
    wait(1);
    gprsSerial.printf("\r\nATD%s;\r\n",NULL==number?phoneNumber:number);
    return 0;
}

int GSM::answer(void)
{
    gprsSerial.printf("ATA\r\n");
    return 0;
}

int GSM::join(char* apn, char* userName, char* passWord)
{
    char gprsBuffer[64];
    int ret = 0;

    cleanBuffer(gprsBuffer,64);
    snprintf(gprsBuffer,sizeof(gprsBuffer),"AT+CGSOCKCONT=1,\"IP\",\"%s\"\r\n", apn);
    if(0 != sendCmdAndWaitForResp(gprsBuffer, "OK", DEFAULT_TIMEOUT)) {
        ret = -1;
        goto failure;
    }

    cleanBuffer(gprsBuffer,64);
    snprintf(gprsBuffer,sizeof(gprsBuffer),"AT+CSOCKAUTH=1,1,\"%s\",\"%s\"\r\n", passWord, userName);
    if(0 != sendCmdAndWaitForResp(gprsBuffer, "OK", DEFAULT_TIMEOUT)) {
        ret = -1;
        goto failure;
    }

    if(0 != sendCmdAndWaitForResp("AT+CSOCKSETPN=1\r\n", "OK", DEFAULT_TIMEOUT)) {
        ret = -1;
        goto failure;
    }
    
    if(0 != sendCmdAndWaitForResp("AT+CIPMODE=0\r\n", "OK", DEFAULT_TIMEOUT)) {
        ret = -1;
        goto failure;
    }

    if(0 != sendCmdAndWaitForResp("AT+NETOPEN\r\n", "OK", DEFAULT_TIMEOUT)) {
        ret = -1;
        goto failure;
    }

    failure:
    return ret;

}

int GSM::SetBlocking(int netopen_to, int cipopen_to, int cipsend_to)
{
    int ret = 0;
    char gprsBuffer[64];

    cleanBuffer(gprsBuffer,64);
    snprintf(gprsBuffer,sizeof(gprsBuffer),"AT+CIPTIMEOUT=%d,%d,%d\r\n", netopen_to, cipopen_to, cipsend_to);
    if(0 != sendCmdAndWaitForResp(gprsBuffer, "OK", DEFAULT_TIMEOUT)) {
        ret = -1;
        goto failure;
    }

    failure:
    return ret;
}

int GSM::connectTCP(char *ip, int port)
{
    int ret = 0;
    char cipstart[64];
#if 0
    if(0 != sendCmdAndWaitForResp("AT+CIPOPEN=0,\"TCP\",\"\",\r\n", "OK", 5)) {
        ret = -1;
        goto failure;
    }
#endif
    sprintf(cipstart, "AT+CIPOPEN=0,\"TCP\",\"%s\",%d\r\n", ip, port);
    if(0 != sendCmdAndWaitForResp(cipstart, "OK", DEFAULT_TIMEOUT)) {
        ret = -1;
        goto failure;
    }

    purge();

    failure:
    return ret;
}

int GSM::sendTCPData(char *data, int len)
{
    int ret = 0;
    char cmd[64];

    snprintf(cmd,sizeof(cmd),"AT+CIPSEND=0,%d\r\n",len);
    if(0 != sendCmdAndWaitForResp(cmd,">",DEFAULT_TIMEOUT)) {
        ret = -1;
        goto failure;
    }
    
    if(0 != sendCmdAndWaitForResp(data,"OK",DEFAULT_TIMEOUT)) {
        ret = -1;
        goto failure;
    }

    failure:
    return ret;
}

int GSM::receivedTCPData(char *buff, int len)
{
    readBuffer(buff, len);
    
    if(strlen(buff) <= 0) return -1;
    
    return 0;
}

int GSM::closeTCP(void)
{
    if(0 != sendCmdAndWaitForResp("AT+CIPCLOSE\r\n","OK",DEFAULT_TIMEOUT)) {
        return -1;
    }
    return 0;
}

int GSM::disconnect(void)
{
    if(0 != sendCmdAndWaitForResp("AT+NETCLOSE\r\n","OK",DEFAULT_TIMEOUT)) {
        return -1;
    }
    return 0;
}
