#pragma once

#include "mbed.h"
#include <sstream>

#define _SerialBase SerialBase

extern Serial dbg;

class PSUSer
{
public:
    
    /*  Constructor 
        
        \param Tx is the serial ports transmit pin (board to PSU)
        \param Rx is the serial ports receive pin (PSU to board)
        \param baudrate the baudrate of the modem use default:9600
    */
    PSUSer(PinName Tx, PinName Rx, unsigned int baudrate = 9600);
    ~PSUSer();
    
    bool readable();
    int get();
    
    enum { 
        // waitFinalResp Responses
        NOT_FOUND     =  0,
        WAIT          = -1, // TIMEOUT
        RESP_OK       = -2, 
        RESP_ERROR    = -3,
        RESP_PROMPT   = -4,
        
        // special timout constant
        TIMEOUT_BLOCKING = -1
    };
    
    
    /** Write formated date to the physical interface (printf style)
        \param fmt the format string
        \param .. variable arguments to be formated
        \return bytes written
    */
    void sendFormated(const char* format);
    
    void getline(char *buffer);
    /*  Check the device is ready or not
        
        \param PV is PSU ID. 06 is default PSU ID
    */
    bool init(const char *n = "06");
    
    /*  send PV for setting constant output Voltage
        
        \param PV is value for output voltage
    */
    bool sendPV(float PV);
    
    /*  send PC for setting constant output Current
        
        \param PC is value for output voltage
    */
    bool sendPC(float PC);
    
    /*  read Voltage's setting or read PV
    */
    bool readPV(float *val);
    
    /*  read Current's setting or read PC
    */
    bool readPC(float *val);
    
    /*  read voltage ouput now
    */
    bool readMV(float *val);
    
    /*  read current ouput now
    */
    bool readMC(float *val);
    
    /*  on/off the output of power supply
    */
    bool power(unsigned int PWR);
    
    /* clear the pending input data */
    virtual void purge(void) 
    { 
        while (readable())
            get();
    }
    
    void _send(const void* buf, int len);
    
    string _getline();
    
    typedef int (*_CALLBACKPTR)(const char* buf, int len, void* param);
    
    int waitResp(_CALLBACKPTR cb = NULL, void *param = NULL, int timeout_ms = 10000); 
    
    template<class T>
    inline int waitResp(int (*cb)(const char* buf, int len, T* param), 
                    T* param, 
                    int timeout_ms = 10000) 
    {
        return waitResp((_CALLBACKPTR)cb, (void*)param, timeout_ms);
    }
    
private:    
    Serial PSU_ser;  
    bool ok;
    
protected:
    
    virtual void wait_ms(int ms)   { if (ms) ::wait_ms(ms); }
    
    typedef struct {float *volt;} VParam;
    static int _cbVolt(const char* buf, int len, VParam *param);
    typedef struct {float *ampere;} AParam;
    static int _cbAmpe(const char* buf, int len, AParam *param);
};


// -----------------------------------------------------------------------

#ifdef RTOS_H
/** Use this template to override the lock and wait functions of the 
    modem driver in a Rtos system. For example declare it the modem 
    object as MDMRtos<MDMSerial> instead of MDMSerial.
*/
template <class T>
class MDMRtos :  public T
{
protected:
    //! we assume that the modem runs in a thread so we yield when waiting
    virtual void wait_ms(int ms)   {
        if (ms) Thread::wait(ms);
        else    Thread::yield();
    }
    //! lock a mutex when accessing the modem
    virtual void lock(void)     { _mtx.lock(); }  
    //! unlock the modem when done accessing it
    virtual void unlock(void)   { _mtx.unlock(); }
    // the mutex resource
    Mutex _mtx;
};
#endif