#include "psu.h"
#include <sstream>

#define OK      0
#define Vmax    60  // maximun voltage to 60 Volt DC
#define Imax    40  // maximun current to 5 Ampere

#define MAX_SIZE        128   //!< max expected messages
//! check for timeout
#define TIMEOUT(t, ms)  ((ms != TIMEOUT_BLOCKING) && (ms < t.read_ms())) 

PSUSer::PSUSer(PinName Tx, PinName Rx, unsigned int baudrate) : PSU_ser(Tx, Rx)
{
    PSU_ser.baud(baudrate);
    PSU_ser.format(8,SerialBase::None,1);
    //PSU_ser.attach(this, &PSUSer::getline, Serial::RxIrq);
}

PSUSer::~PSUSer()
{
    
}

void PSUSer::sendFormated(const char* format)
{
    string frtm;
    frtm += format;
    //dbg.printf("\r\nsend: format\t%s",frtm.c_str());
    _send(frtm.c_str(), frtm.size());
}

void PSUSer::getline(char* buffer)
{
    string result = _getline();
    strcpy(buffer,result.c_str());
}

int PSUSer::waitResp(_CALLBACKPTR cb, void *param, int timeout_ms)
{
    char buf[MAX_SIZE + 64 /* add some more space for framing */]; 
    Timer timer;
    timer.start();
    do {
        getline(buf);
        //dbg.printf("\r\nread:\tbuf\t%s\r\n",buf);
        if(cb)
        {
            int len = strlen(buf);
            int ret = cb(buf, len, param);
            if(WAIT != ret)
                return ret;
        }
        
        if(strstr(buf, "OK"))
            return RESP_OK;
        
        wait_ms(10);
    }
    while(!TIMEOUT(timer, timeout_ms));
    
    return WAIT;
}

bool PSUSer::init(const char *n)
{
    ok = false;
    char snd[10];
    sprintf(snd,"ADR %s\r",n);
    sendFormated("RST\r");
    wait_ms(100);
    sendFormated(snd);
    ok = (RESP_OK == waitResp(NULL,NULL,2000));
    return ok;
}

bool PSUSer::sendPV(float PV)
{
    ok = false;    
    if(PV > Vmax)
        PV = Vmax;
    char snd[10];
    sprintf(snd,"PV %.2f\r",PV);
    sendFormated(snd);
    ok = (RESP_OK == waitResp(NULL,NULL,1000));
    return ok;
}
    
bool PSUSer::sendPC(float PC)
{
    ok = false;    
    if(PC > Imax)
        PC = Imax;
    char snd[10];
    sprintf(snd,"PC %.2f\r",PC);
    sendFormated(snd);
    ok = (RESP_OK == waitResp(NULL,NULL,1000));
    return ok;
}

int PSUSer::_cbVolt(const char* buf, int len, VParam *param)
{
    if(param)
    {
        if (sscanf(buf, " %f",param->volt)==1)
            return RESP_OK;
        else return WAIT;
    }
    else
        return WAIT;
}

int PSUSer::_cbAmpe(const char* buf, int len, AParam *param)
{
    if(param)
    {
        if (sscanf(buf, " %f",param->ampere)==1)
            return RESP_OK;
        else return WAIT;
    }
    else
        return WAIT;
}

bool PSUSer::readPV(float *val)
{
    ok = false;
    VParam param;
    param.volt = val;
    sendFormated("PV?\r");
    ok = (RESP_OK == waitResp(_cbVolt,&param));
    return ok;
}

bool PSUSer::readPC(float *val)
{
    ok = false;    
    AParam param;
    param.ampere = val;
    sendFormated("PC?\r");
    ok = (RESP_OK == waitResp(_cbAmpe,&param));
    return ok;
}

bool PSUSer::readMV(float *val)
{
    purge();
    ok = false;
    VParam param;
    param.volt = val;
    sendFormated("MV?\r");
    ok = (RESP_OK == waitResp(_cbVolt,&param));
    return ok;
}

bool PSUSer::readMC(float *val)
{
    purge();
    ok = false;
    AParam param;
    param.ampere = val;
    sendFormated("MC?\r");
    ok = (RESP_OK == waitResp(_cbAmpe,&param));
    return ok;
}

bool PSUSer::power(unsigned int pwr)
{
    ok = false;    
    if(pwr > 1)
        pwr = 1;
    char snd[10];
    sprintf(snd,"OUT %d\r",pwr);
    sendFormated(snd);
    ok = (RESP_OK == waitResp(NULL,NULL,1000));
    return ok;
}

void PSUSer::_send(const void* buf, int len)
{
    const char * kirim = (const char*)buf;
    for(int i=0; i<len; i++)
    {
        PSU_ser.putc(kirim[i]);
    }
}

string PSUSer::_getline()
{
    string result;
    while(PSU_ser.readable())
        result += PSU_ser.getc();
    return result;
}

bool PSUSer::readable()
{
    return PSU_ser.readable();
}

int PSUSer::get()
{
    if(!PSU_ser.readable())
        return EOF;
    return PSU_ser.getc();
}