#include "mbed.h"
#include "TextLCD.h"
#include "PinDetect.h"
#include "Menu.h"
#include "Comms.h"
#include "PCComms.h"
#include "CANComms.h"
#include "Gears.h"
#include "LEDS.h"
#include "bigchar.h"
//Test comment 2
//Refresh interval for LCD
#define LCD_REFRESH_TIME 150000

//Refresh interval for rev LEDs
#define REV_REFRESH_TIME 50000

//Warning lights
DigitalOut warn[] = { (p20), (p19), (p27), (p18) };

//mBED LEDs
DigitalOut debug[] = { (LED1), (LED2), (LED3) };

//Heartbeat LED
DigitalOut heartbeat(LED4);

//Rev LEDS
PwmOut leds[] = { (p24), (p25), (p26), (p23), (p22), (p21) };

//LCD
TextLCD lcd(p5, p6, p7, p8, p9, p10, p11);

//Tickers for refreshing
Ticker lcdRefreshTicker;
Ticker revRefreshTicker;

//Used to animate LEDs for testing
Ticker increment;

//Main car state structure
State car;

//Classes for various parts of the firmware
Menu dashMenu(&lcd, p16, p17, p12); //*LCD, OK, Left, Right
PCComms pc(&car);
CANComms *can;
Gears gearButtons(p14, p13, p15, &car.gear, &pc); //Up, Neutral, Down, *Current Gear
LEDS revs(leds);

//Refresh the rev LEDs and warning LEDs
void revRefresh()
{
    revs.refresh(car.rpm);

    if(car.voltage < 12) {
        warn[0] = 1;
    } else {
        warn[0] = 0;
    }
    
    if(car.coolant_temp > 110) {
        warn[1] = 1;
    } else {
        warn[1] = 0;
    }
    
    if(car.rpm == 0 and car.gear != 0)
    {
        warn[2] = 1;
    } else {
        warn[2] = 0;
    }
}

//Refresh the LCD
void lcdRefresh()
{
    //If menu off screen, display HUD
    if(dashMenu.display == false) {
        lcd.locate(0, 0);
        lcd.printf("R:%-5.1d S:%-3d", car.rpm, car.speed);
        lcd.locate(0, 1);
        lcd.printf("%-4.1fV %-4.1f\xDF\x43", car.voltage, car.coolant_temp);

        write_bigchar(&lcd, 13, car.gear);
    }
    //Otherwise show menu
    else {
        dashMenu.refresh();
    }

    //Blink heartbeat
    heartbeat = !heartbeat;
}


//Used to animate LEDs for testing
/*void doIncrement()
{
    if(car.rpm < LIMIT && car.gear > 0)
        car.rpm++;
}*/

//Startup animation
void selfTest()
{
    lcd.printf("    FBR 2012");
    lcd.locate(0,1);
    lcd.printf(" Ready to drive");

    //Light up LEDs
    for(int i = 0; i < LEDS::NUM_LEDS; i++) {
        leds[i] = true;
        if(i < 4)
            warn[i] = true;
        wait(0.1);
    }

    //Turn off LEDs
    for(int i = LEDS::NUM_LEDS - 1; i >= 0; i--) {
        leds[i] = false;
        if(i < 4)
            warn[i] = false;
        wait(0.1);
    }

    lcd.cls();
}

int main()
{
    printf("\n");
    printf("FBRDash\n");

    //Initialise state
    car.rpm = 0;
    car.gear = 2;
    car.speed = 0;
    car.coolant_temp = 0;
    car.throttle_pos = 0;
    car.manifold_pres = 0;
    car.air_temp = 0;
    car.lambda = 0;
    car.accel_x = 0;
    car.accel_y = 0;
    car.oil_temp = 0;
    car.voltage = 10.0;

    //Set up menu
    dashMenu.addItem<float>("Coolant Temp  ", "%12.1f\xDF\x43", &car.coolant_temp); // \xDF\x43 -> &#65533;C . Need code for C as otherwise it gets taken as hex digit.
    dashMenu.addItem<unsigned char>("Air Temp      ", "%12d\xDF\x43", &car.air_temp);
    dashMenu.addItem<float>("Throttle Pos  ", "%12.1fd\xDF", &car.throttle_pos);
    dashMenu.addItem<unsigned char>("Manifold Pres ", "%10d psi", &car.manifold_pres);
    dashMenu.addItem<unsigned char>("Lambda        ", "%14d", &car.lambda);
    dashMenu.addItem<unsigned char>("Oil Temp      ", "%12d\xDF\x43", &car.oil_temp);

    //Set up characters on LCS
    setup_bigchar(&lcd);

    //Do bootup animation
    selfTest();

    //Start refresh tickers
    lcdRefreshTicker.attach_us(&lcdRefresh, LCD_REFRESH_TIME);
    revRefreshTicker.attach_us(&revRefresh, REV_REFRESH_TIME);
    //increment.attach(&doIncrement, 0.0005);

    can = new CANComms(&car);
    
    //Infinite loop - program is interrupt driven
    while(true) {
        __WFI();
    }
}