#include "Menu.h"
#include "mbed.h"
#include <string>

//Drive the menu system

//Initialise and attach button handlers
Menu::Menu(TextLCD* _screen, PinName ok, PinName left, PinName right)
{
    screen = _screen;
    
    entryCount = 0;
    
    position = 0;
    
    btnOK = new PinDetect(ok, PullUp);
    btnLeft = new PinDetect(left, PullUp);
    btnRight = new PinDetect(right, PullUp);
    
    btnOK->attach_deasserted(this, &Menu::enter);
    btnOK->attach_asserted_held(this, &Menu::done);

    btnLeft->attach_asserted(this, &Menu::left);
    btnLeft->attach_asserted_held(this, &Menu::leftHeld);
    btnLeft->attach_deasserted(this, &Menu::cancelLeftHeld);
    
    btnRight->attach_asserted(this, &Menu::right);
    btnRight->attach_asserted_held(this, &Menu::rightHeld);
    btnRight->attach_deasserted(this, &Menu::cancelRightHeld);

    btnOK->setAssertValue(0);
    btnLeft->setAssertValue(0);
    btnRight->setAssertValue(0);

    btnOK->setSampleFrequency();
    btnLeft->setSampleFrequency();
    btnRight->setSampleFrequency();
    
    ignoreNextEnter = false;
}

void Menu::refresh()
{
    //Create chars (if needed) for left and right arrows and editable indicator
    char labelLeft = (!edit & position > 0)?leftArrow:' ';
    char labelRight = (!edit & position < entryCount)?rightArrow:' ';
    char editIndic = (entries[position]->editable)?'*':' ';

    screen->locate(0, 0);
    
    //If position is on an entry, display it
    if(position <= entryCount - 1)
    {
        screen->printf("%c%-14s%c", labelLeft, (entries[position])->label.c_str(), labelRight);
        
        char editLeft = (edit)?'-':editIndic;
        char editRight = (edit)?'+':' ';
        
        screen->putc(editLeft);
        screen->printf(entries[position]->getFormatted().c_str());
        screen->locate(15, 1);
        screen->putc(editRight);
    }
    //Otherwise show the return command
    else
    {    
        screen->printf("%cReturn                         ", leftArrow);
    }
}

//Handler for enter button
void Menu::enter()
{
    //Menu not on screen, display it
    if(!display && !ignoreNextEnter)
    {   
        display = true;
        edit = false;
    }
    //Menu on screen
    else
    {
        //On an entry
        if(position <= entryCount - 1)
        {
            //If its editable, toggle edit status
            if(entries[position]->editable)
                edit = !edit;
        }
        //On return
        else
        {
            //Return, resetting position first
            position = 0;
            done();
        }
    }
    
    ignoreNextEnter = false;
}

//Handler for enter held, clear the menu retaining position
void Menu::done()
{
    //Hide the menu
    display = false;
    //TODO: Can't remember why I needed this
    ignoreNextEnter = true;
}

//Handler for left button
void Menu::left()
{
    //If not editing, scroll
    if(!edit && display && position > 0)
        position--;
    //If editing, edit
    else if(edit)
        entries[position]->decrement();
}

//Handler for left button held, start scrolling through menu.
void Menu::leftHeld()
{
    left();
    leftHeldTimeout.attach(this, &Menu::leftHeld, holdRepeatTime);
}

//Handler for left button release, stop scrolling
void Menu::cancelLeftHeld()
{
    leftHeldTimeout.detach();
}

//Same as left
void Menu::right()
{
    if(!edit && display && position < entryCount)
        position++;
    else if(edit)
        entries[position]->increment();
}

//Same as left
void Menu::rightHeld()
{
    right();
    rightHeldTimeout.attach(this, &Menu::rightHeld, holdRepeatTime);
}

//Same as left
void Menu::cancelRightHeld()
{
    rightHeldTimeout.detach();
}