#include "mbed.h"
#include "ShootingSystem.h"
#include "Command.h"
#include "I2CServo.h"
#include "Shooter.h"
#include "AmmoPusher.h"
#include "AmmoSupplier.h"

bool ShootingSystem::mIsShootable = false;

ShootingSystem::ShootingSystem(
        I2CServo* angleManagerServo, AmmoPusher* ammoPusher,
        AmmoSupplier* ammoSupplier, Shooter* shooter,
        I2CServo* positionManager ){
    
    mShooterAngleServo  = angleManagerServo;
    mShooter            = shooter;
    mAmmoPusher         = ammoPusher;
    mAmmoSupplier       = ammoSupplier;
    mPositionManager    = positionManager;
    mState              = SHOOT_PREPARATION;
}

void ShootingSystem::update( Command command ){
    bool isChangeable = ( mState != SHOOTING );
    isChangeable &= ( mState != SHOOT_PREPARATION );
    
    if ( isChangeable ){
        // 発射中でないなら角度と電圧が変更可能
        mShooterAngleServo->setTargetPosition( command.getShootingAngleAnalog() );
        mPositionManager->setTargetPosition( command.getShooterPosition() );
        mShooter->setVoltage( command.getShootVoltage() );
    }
    
    mIsShootable = ( mState == WAITING );
    
    switch ( mState ){
        case WAITING:
            waiting( command );
            break;
            
        case SHOOTING:
            shooting();
            break;
            
        case SHOOT_WAITING:
            shootWaiting();
            break;
            
        case SHOOT_PREPARATION:
            shootPreparation();
            break;
            
        case AMMO_SUPPLYING:
            ammoSupply();
            break;
            
        default:
            break;
    }
}

ShootingSystem::~ShootingSystem(){
    delete mShooterAngleServo;
    delete mAmmoPusher;
    delete mAmmoSupplier;
    delete mShooter;
    delete mPositionManager;
}

void ShootingSystem::waiting( Command command ){
    if ( command.isSupplying() ){
        mAmmoSupplier->supply();
        mState = AMMO_SUPPLYING;
    } else if ( command.isShooting() ){
        mState = SHOOTING;
    }
}

void ShootingSystem::shooting(){
    mShooter->launch();
    mAmmoPusher->push();
    
    if ( mAmmoPusher->hasPushed() ){
        mState = SHOOT_WAITING;
        mTimer.reset();
        mTimer.start();
    }
}

void ShootingSystem::shootWaiting(){
    mAmmoPusher->draw();
    
    if ( mTimer.read() > 0.7f ){
        mShooter->stop();
        mState = SHOOT_PREPARATION;
    }
}

void ShootingSystem::shootPreparation(){
    if ( mAmmoPusher->hasDrawn() ){
        mState = WAITING;
    }
}

void ShootingSystem::ammoSupply(){
    if ( mAmmoSupplier->hasSupplied() ){
        mState = WAITING;
    }
}
