#include "mbed.h"
#include "SerialToI2C.h"

SerialToI2C* SerialToI2C::mInstance  = NULL;
const char SerialToI2C::mControlMask = 0x30;
const char SerialToI2C::mDetailMask  = 0x0F;
const char SerialToI2C::mDutyMask    = 0xC0;

const PinName SerialToI2C::mAddressDeciderPinName[] = {
    dp17, dp18, dp26, dp25
};

SerialToI2C::SerialToI2C( PinName sda, PinName scl, PinName tx, PinName rx ){
    mMoveDirection  = 0;
    mMoveCoeff      = 0.0f;
    mRoll           = 0.0f;
    mActionType     = STOP;
    mRecv           = OWN_POLE | STOP;
    mI2CSentData    = mRecv;
    mIsI2CSentDataChangeable = true;
    mAimState       = OWN_POLE;
    mStopDetailType = HIGH_SPEED;
    
    mLED = new DigitalOut( LED1, true );
    
    mAddress = 0xA0;    // 10100000(2)
    for ( int i = 0; i < 4; ++i ){
        // 101xxxx0 --- xxxxの部分に入力データが入る
        DigitalIn d( mAddressDeciderPinName[ i ] );
        mAddress |= d.read() << ( i + 1 );
    }
    
    mI2C  = new I2CSlave( sda, scl );
    mI2C->address( mAddress );
    
    mXBee = new RawSerial( tx, rx );
    mXBee->baud( 19200 );
    
    mXBee->attach( this, &SerialToI2C::recvSerial, Serial::RxIrq );
}

void SerialToI2C::update(){
    stringToI2CData();
    updateI2CSlave();
}

void SerialToI2C::updateI2CSlave(){
    switch ( mI2C->receive() ){
        case I2CSlave::ReadAddressed:{
            char buf[] = { mActionType, 0, mAimState, mIsSupplying };
            
            switch ( mActionType ){
                case STOP:
                    buf[ 1 ] = mStopDetailType;
                    break;
                    
                case MOVE:
                    buf[ 1 ] = mMoveDirection;
                    break;
                    
                case ROLL:
                    buf[ 1 ] = mRoll;
                    break;
                    
                case SHOOT:
                    break;
                    
                default:
                    mActionType = STOP;
                    break;
            }
            
            mI2C->write( buf, 4 );
            
            mI2CSentData = mRecv;
            break;
        }
        case I2CSlave::WriteGeneral:
            break;
        
        case I2CSlave::WriteAddressed:
            break;
            
        case I2CSlave::NoData:
            break;
    };
}

void SerialToI2C::stringToI2CData(){
    // 受信したデータをロボットの行動情報に変換
    mActionType     = static_cast< ActionType >( mRecv & 0x18 );    // 00011000(2)
    char detail     = mRecv & 0x07; // 00000111(2)
    mAimState       = static_cast< AimState >( mRecv & 0xC0 );  // 11000000(2)
    mIsSupplying    = mRecv & 0x20; // 00100000(2)
    
    switch ( mActionType ){
        case MOVE:{
            mMoveDirection = detail;
            break;
        }
        case ROLL:
            mRoll = detail;
            break;
            
        case STOP:
            mStopDetailType = static_cast< StopDetailType >( detail );
            break;
            
        case SHOOT:
            break;
            
        default:
            break;
    }
}

void SerialToI2C::recvSerial(){
    char recv = mXBee->getc();
    
    if ( ( recv != mRecv ) &&
         ( mI2CSentData == mRecv ) ){
        mRecv = recv;
    }
}
