// *********************************************************************
//
// Display driver for Solomon SSD2119 controller connected to mbed board
// with simple terminal program
//
// ** szdiditalsquare (eBay) display connected on SPI + control pins  **
//  mbed            SSD2219 (on szdigitalsquare board)
//  p5 -mosi        (28) SDA
//  p6 -miso         NC
//  p7 -sck         (26) SCL
//  p8 -cs          (19) CS
//  p15-res         (23) RESET
//  p16-c/d         (20) RS
//  p17-rd          (22) RD
//  p18-wr          (21) WR
//
// Pins PS0 - PS3 should be open on system board = 4bit SPI mode
//
// After sucessful init you'll see color bars on whole display and
// welcome message. Then you can start typing in your serial terminal
// and information will appear on the display
//
// 28.11.2010 port from CrystalFonts & Nokia LCD lib by Imrich Konkol
// 29.11.2010 ZX Spectrum SCR viewer added
//            - to show test.scr from internal mbed drive type \\ at
//              PC terminal prompt
// 01.12.2010 bright fixed, slideshow added, terminal session start
//            after slideshow finish
// 04.12.2010 fixed non bright values (3/4 of intensity instead of 1/2)
// 08.12.2010 added ULA+ format for slideshow (6912 + 64 bytes palette)
// *********************************************************************
#include "mbed.h"
#include "EthernetNetIf.h"
#include "HTTPClient.h"
#include "Display.h"

LocalFileSystem local("local");

/* SPI definitions moved to Display.c module
SPI spi(p5, p6, p7); // mosi, miso, sclk
DigitalOut DCS(p8);     // Display chip select
DigitalOut DRS(p15);   // Display reset
DigitalOut DCD(p16);   // Display command/data
DigitalOut DRD(p17);   // Display read
DigitalOut DWR(p18);   // Display write
*/
DigitalOut myled(LED1);

Serial pc(USBTX, USBRX);    //debug info - see raw data being transferred back and forth as GBA is booting

EthernetNetIf eth;
HTTPClient http;

/*-----------------------------------------------------------------------------
                                      Global Variables
-----------------------------------------------------------------------------*/
extern uint16_t FOREGROUND, BACKGROUND;     // Display back and foreground - defined in Display.h
int ShowType = 0;                           // What images to display: 0 - local, 1 - internet

unsigned char cat(char *filename) {
    int c;
    unsigned int x,y;

    printf("Opening File...\n"); // Drive should be marked as removed
    FILE *fp = fopen(filename, "r");
    if (!fp) {
        printf("File %s could not be opened!\n",filename);
        return(1);
    }
    cls();
    Display_Home();
    x = 0;
    y = 0;

    while ( ( c=fgetc(fp) ) != EOF ) {

        LcdGotoXY(x,y);

        if (c == 13) {
            y++;
            x=0;
        };
        if (c == 8)  {
            x--;
        };

        if (c > 31) LcdChr(c);
        x++;
        if (x>39) {
            y++;
            x=0;
            if (y>39) y=0;
        }
    }
    fclose(fp);
    return(0);
}



void terminal() 
{
    int x, y, r=1;
    char c, xc=' ';
    
        FOREGROUND = GREEN;
        BACKGROUND = BLACK;
        display_rgb(BACKGROUND);

        LcdGotoXY(4,1);
        LcdStr("Below you can see what you write");
        LcdGotoXY(12,2);
        LcdStr("in your PC terminal");
        
        x = 0;
        y = 3;
        while (r == 1) 
        {
            LcdGotoXY(x,y);
            c = pc.getc();
            if (c == 13) {
                y++;
                x=0;
            }; // endif c==13
            
            if (c == 8)  {
                x--;
            }; // endif c==8

            if (c > 31) {
                LcdChr(c);
                if ( (xc == '\\' ) && ( c == '\\' ) ) {
                    //scr2lcd("test.scr");
                    //LCD_test();
                    //SldShw();
                    r=0;
                }

                xc = c;
            } // endif c>31
            x++;
            if (x>39) {
                y++;
                x=0;
                if (y>39) y=0;
            } // endif x>39
        } // endwhile
    
}

// Always print system messages on certain line of display
void SysMsg(char *dataPtr) {

        LcdGotoXY8(0,29);
        LcdStr8x8( dataPtr );
}        


// Main loop starts here
int main() {

    // Variable definitions
    uint8_t y;
    //unsigned int i;
    char str[40];
    char err[40];       // Error message string
    char url[256];       // Remote URL
    char fname[80];      // Local filename
    int c,w;
    FILE *fl;
    unsigned int i, j, a, rnd, last, blast;
    const int num_files=300;
    int names[num_files];
    int lenghts[num_files];

    // Variable initialization
    a = 0;
    i = 0;
    last = 0;
    blast = 0;
   
    //Configure the Fastest Baud Rate
    pc.baud(115200);
    printf("Program start.\n");

    myled = 0;
    initialization();       // Initialize LCD

    printf("SPI and display initialized.\n");

    LCD_test();             // Draw border
    printf("Display test.\n");

    wait(1.0);
    if (myled == 0) {
        myled = 1;
    } else {
        myled = 0;
    }

    // Show welcome message
    Display_Home();
    LcdGotoXY8(12,10);
    LcdStr8x8("Welcome to mBed");
    LcdGotoXY8(5,11);
    LcdStr8x8("Retro Computing Picture Frame");
    LcdGotoXY8(0,13);
    LcdStr8x8("Based on SSD2119 controller& TFT 320x240");     
    
    SysMsg("Configuring Ethernet...");
    
    // Configure ethernet interface    
    EthernetErr ethErr = eth.setup();
    if(ethErr) {
        sprintf(err,"Error %d in ETH setup.", ethErr);
        SysMsg(err);
        ShowType = 0;               // Local slideshow only
    } else {
        IpAddr ip = eth.getIp();
        sprintf(err, "Setup OK, local IP: %d.%d.%d.%d", ip[0], ip[1], ip[2], ip[3]);
        SysMsg(err);
        ShowType = 1;               // Ethernet available - we can download pictures
    }

    // Do some common stuff for both ShowType modes
    LCD_test();         // Show colorfull border again    
    wait(2.0);
    myled = 0;
    Display_Home();
    LcdGotoXY(13,0);
    LcdChr('H');
    LcdGotoXY(14,0);
    LcdChr('e');
    LcdGotoXY(15,0);
    LcdChr('l');
    LcdGotoXY(16,0);
    LcdChr('l');
    LcdGotoXY(17,0);
    LcdChr('o');

    LcdStr(" mBed World!");

    y = 1;
    LcdGotoXY(0,y);
    LcdStr("Local filesystem contains these files:");
    y++;
    LcdGotoXY(0,y);
    Dir();                  // Show directory listing

    wait(1.0);

    ShowPalette256();        // Show actual 256 colors palette

    for (i=3;i>0;i--) 
    {
        LcdGotoXY(0,38);
        sprintf(str, "Displaying images in %d seconds...", i);
        switch (i) {
            case 1:
                strcat(str,"/");
                break;
            case 2:
                strcat(str,"|");
                break;
            case 3:
                strcat(str,"\\");
                break;
            case 4:
                strcat(str,"-");
                break;
            default:
                break;
        }

        LcdStr(str);
        wait(1.0);
    }
    
    
    // Check if we have the list of files
    if ( (fl = fopen("/local/scrnpath.txt","r") ) == NULL ) {              // Fl = FileList
      printf("Unable to open file list! Fallback to local.\n");
      ShowType = 0; // Local
    } else {
        while ( ( ( c = fgetc(fl) ) != EOF ) && ( a < num_files) ) {
        i++;
        if ( c == '\n') {
            last = i;
            if ( a == 0 ) {
                names[a] = 0;
                lenghts[a] = last;
            } else {
                names[a] = blast;
                lenghts[a] = last - blast;
            } // If2
            blast = last;
            a++;
        } // If2

        } // while
    
    } // Else
    // Variable 'a' now holds number of pictures available

    srand ( 99 );
    LCD_test();                                 // Restore rainbow background    
    
    // Now distinguish between both modes and do different things
    if ( ShowType == 0 ) { 
        while (1) {
            SldShw();                                   // Slide show from ZX screens
            
            if ( pc.readable() ) terminal();               // If key pressed start terminal
            
            wait(3.0);
            } // loop forever
        
        } else {
       
       while (1) { 
            // We do picture show from the net
            for (j=0; j < 48; j++ ) {
            // Pseudo Random Picture file selection
            rnd = rand() % a;

            printf("Picture[%d] Names[%d]   Lenghts[%d]\n", rnd, names[rnd], lenghts[rnd]);
            printf("Random file: %d\n", rnd);
            fseek(fl, names[rnd], SEEK_SET);

            w = fgetc(fl);

            sprintf(url, "http://wos.meulie.net/pub/spectrum/screens/load/%c/scr/%c", w+32, w);

            for (i=2; i<lenghts[rnd]; i++) {
                c = fgetc(fl);
                sprintf(url, "%s%c", url, c);
            }
            printf("URL: %s\n",url);
   
            // Open a file to write.
            sprintf(fname,"/local/%d.scr",j);
            printf("Local file name: %s\n", fname);
            HTTPFile file(fname);


            // Insert the search term into the URL
            //sprintf(url, "http://wos.meulie.net/pub/spectrum/screens/load/a/scr/ACZGeneralLedger2000.scr");

            // Request a page and store it into a file.
            HTTPResult r = http.get(url, &file);

            if(r==HTTP_OK)
            {
                pc.printf("File fetched OK.\n"); //:\"%s\"\n", file.gets()); 
            } else {
                pc.printf("Error fetching file %d\n", r);
            }
            
            sprintf(fname,"%d.scr",j);
            scr2lcd(fname);
            wait(2.0);
            
            } // For (j)

       } // Loop Forever
        
       fclose(fl); // Close index file
        
       } // Else (ShowType)
    
} // End main()

