#ifndef ENEMY_H
#define ENEMY_H

#include "mbed.h"
#include "N5110.h"
#include "Gamepad.h"
#include <ctime>
#include <cmath>

/** Enemy Class
@brief Library that updates the position of all enemies and enemy weapons for rosen game
@author Ozoemena Adrian Ikrnna
@date 8th May 2019
*/
class Enemy
{
    public:
    /** constructor */
    Enemy();
    /** destructor */
    ~Enemy();
    
    /** A mutator method that initializes the number of enemies
    *@param noShooters the number of enemy shooters on screen
    *@param noSeekers the number of enemy seekers on screen
    *@<c>{note that these should not be confused with seno and shno which are used to
    *reference particular enemies i.e shooter1 and shooter2}</c>
    */
    void init(int noShooters, int noSeekers);
    /** A mutator method that sets the number of shooters, used for increasing game difficulty
    *@param noShooters the number of enemy shooters on screen
    */
    void set_noshooters(int noShooters);
    /** A mutator method that sets the number of seekers, used for increasing game difficulty
    *@param noSeekers the number of enemy seekers on screen
    */
    void set_noseekers(int noSeekers);
    /** Draws the shooter on screen
    *@param &lcd pointer to the N5110 library used for the lcd display
    */
    void draw_shooter(N5110 &lcd);
    /** Draws the seeker on screen
    *@param &lcd pointer to the N5110 library used for the lcd display
    */
    void draw_seeker(N5110 &lcd);
    /** A mutator method that updates the shooter position according to the player ships position
    *@param ship_xpos the ships x co-ordinate
    *@param ship_ypos the ships y co-ordinate
    */
    void update_shooter(int ship_xpos, int ship_ypos);
    /** A mutator method that updates the seekers position according to the player ships position
    *@param ship_xpos the ships x co-ordinate
    *@param ship_ypos the ships y co-ordinate
    */
    void update_seeker(int ship_xpos, int ship_ypos);
    /** resets the shooter to the top of screen
    *@param shno the shooter number used to reference the shooter of interest
    *Ranges from 0-3 as there can be a maximum of three shooters
    */
    void reset_shooter(int shno);
    /** resets the seeker to the top of screen
    *@param seno the seeker number used to reference the seekerr of interest
    *Ranges from 0-3 as there can be a maximum of three seekers
    */
    void reset_seeker(int seno);
    /** draws the projectile fired from the shooter
    *@param &lcd pointer to the N5110 library used for the lcd display
    *@param &pad pointer to the gamepad library used to read inputs and send outputs to the gamepad
    */
    void draw_shw(N5110 &lcd,Gamepad &pad);
    /** A mutator method that updates the position of the projectile fired from the shooter
    */
    void update_shw();
    /** An accessor method that gets the position of the projectile fired by the shooter
    *@param shno the shooter number used to reference the shooter of interest
    *Ranges from 0-3 as there can be a maximum of three shooters
    */
    Vector2D get_shwpos(int shno);
    /** An accessor method that gets the position of any shooter on screen
    *@param shno the shooter number used to reference the shooter of interest
    *Ranges from 0-3 as there can be a maximum of three shooters
    */
    Vector2D get_shooterpos(int shno);
    /** An accessor method that gets the position of any seeker on screen
    *@param seno the seeker number used to reference the seekerr of interest
    *Ranges from 0-3 as there can be a maximum of three seekers
    */
    Vector2D get_seekerpos(int seno);
    /** A random number generator that creates a random number that can be scaled
    *appropriately using the <ctime> library
    *@param scale this will determine the maximum output of the function
    *@returns a random number between 0 and scale
    */
    int rand_no(int scale);
    /** A mutator function which changes enemy speed and number as time goes on
    *@param time_elapsed the amount of time that has passed since a player pressed play in seconds
    */
    void sh_scaling(float time_elapsed);
    /** Finds the average distance between two points on the 2D screen
    *@param x1 the x co-ordinate of the first point
    *@param y1 the y co-ordinate of the first point
    *@param x2 the x co-ordinate of the second point
    *@param y2 the y co-ordinate of the second point
    *@returns the distance between the two points
    */
    int distance(int x1, int y1, int x2, int y2);
    /** Finds the closest enemy ship on screen
    *used for the orions weapon tracking
    *@param ship_xpos the ships x co-ordinate
    *@param ship_ypos the ships y co-ordinate
    *@param noShooters the number of enemy shooters on screen
    *@param noSeekers the number of enemy seekers on screen
    */
    Vector2D find_closest(int ship_xpos,int ship_ypos,int noSeekers, int noShooters);

    
    private:
    //_______________Private-Methods__________________________________________
    void draw_shw1(N5110 &lcd,Gamepad &pad);
    void draw_shw2(N5110 &lcd,Gamepad &pad);
    void draw_shw3(N5110 &lcd,Gamepad &pad);
    void update_seeker1(int ship_xpos, int ship_ypos);
    void update_seeker2(int ship_xpos, int ship_ypos);
    void update_seeker3(int ship_xpos, int ship_ypos);
    Vector2D shooter_motion(int shooter_xpos,int shooter_ypos,int ship_xpos, int ship_ypos, int projx, int projy,int vx, int vy);
    
    //_______________Private-Variables__________________________________________
    Vector2D _seekerPos[3];     //a 2D array of all the seekers position values
    Vector2D _shooterPos[3];    //a 2D array of all the shooters position values
    Vector2D _shooterSpeed[3];  //a 2D array of all the shooters speed values
    Vector2D _shooterWPos[3];   //a 2D array of all the shooters-projectile position position values
    Vector2D _shooterWSpeed[3]; //a 2D array of all the shooters-projectile speed position values
    int _reset[3];              //an array or reset values used to reset projectiles from the shooter weapon
    int _noShooters;            //the number of shooters on the screen
    int _noSeekers;             //the number of seekers on the screen
    
};
#endif