#include <mbed.h>
#ifndef __MMCx12xM__
#define __MMCx12xM__

// possible I2C addresses, depending on the chip number
enum
{
  MMCx120M = 0x60,
  MMCx121M = 0x64,
  MMCx122M = 0x68,
  MMCx123M = 0x6C,
};

/* Class: MMCx12xM
 *  Control a Memsic MMC212xM magnetometer over I2C
 *
 * Example:
 * > // MMC2120M at address 0x60
 * >
 * > #include "mbed.h"
 * >
 * > I2C i2c(p28, p27);
 * > MMC212xM memsic1(i2c);
 * >
 * > int main() {
 * >     int data[2];
 * >     memsic1.read_raw_values(data, 2);
 * > }
 */

class MMCx12xM : public Base
{
public:
    // constructor in case you already have an I2C bus instance
    MMCx12xM(I2C &i2c, int address = MMCx120M, const char *name = NULL);
    // use this constructor if the sensor is the only device on the bus 
    MMCx12xM(PinName sda, PinName scl, int address = MMCx120M, const char *name = NULL);
    // send a SET coil command
    bool coil_set();
    // send a RESET coil command
    bool coil_reset();
    // read raw (12-bit) axis values
    bool read_raw_values(int *values, int count = 2);
    // start calibration
    void calibrate_begin();
    // take a single measurement for calibration
    void calibrate_step(int count = 2);
    // finish calibration and calculate offset and sensitivity values
    void calibrate_end();
    // read calibrated (-1.0 .. +1.0) axis values
    bool read_values(float *values, int count = 2);
    virtual ~MMCx12xM();
  
private:
    bool _send_command(int command);
    bool _wait_ready(int command);
    bool _read_axis(int *value, int index = -1);

    // reference to the I2C bus
    I2C *_I2C;
    // sensor slave address
    int _addr;
    // did we create the bus instance? (i.e. we should delete it on destruct)
    bool _own_i2c;
    // calibration values
    int _sensitivity[3];
    int _offset[3];
    // temporaries for calibration
    int _maxvals[3];
    int _minvals[3];
};

#endif