/*
 * Copyright (c) 2015 Keith Cullen.
 * All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR "AS IS" AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/**
 *  @file coap_client.h
 *
 *  @brief Include file for the FreeCoAP client library
 */

#ifndef COAP_CLIENT_H
#define COAP_CLIENT_H

#include <time.h>
#include "coap_msg.h"

#define EBADMSG                     74              /* Not a data message */
#define ETIMEDOUT                   110             /* Connection timed out */
#define ECONNRESET                  104             /* Connection reset by peer */
#define ENOSPC                      28              /* No space left on device */

#define COAP_CLIENT_HOST_BUF_LEN    128             /* Buffer length for host addresses */
#define COAP_CLIENT_PORT_BUF_LEN    8               /* Buffer length for port numbers */

/**
 *  @brief Test result enumeration
 */
typedef enum {FAIL = 0, PASS} test_result_t;

/**
 *  @brief Test data typedef
 */
typedef void *test_data_t;

/**
 *  @brief Test function typedef
 */
typedef test_result_t (*test_func_t)(test_data_t);

/**
 *  @brief Message option test data structure
 */
typedef struct
{
    unsigned num;                                                               /**< Option number */
    unsigned len;                                                               /**< Option length */
    char *val;                                                                  /**< Pointer to a buffer containing the option value */
}
test_coap_client_msg_op_t;

/**
 *  @brief Client test message data structure
 */
typedef struct
{
    coap_msg_type_t type;                                                       /**< Message type */
    unsigned code_class;                                                        /**< Message code class */
    unsigned code_detail;                                                       /**< Message code detail */
    test_coap_client_msg_op_t ops;                                             /**< Array of message option test data structures */
    unsigned num_ops;                                                           /**< Size of the array of message option test data structures */
    char *payload;                                                              /**< Buffer containing the payload */
    size_t payload_len;                                                         /**< Length of the buffer containing the payload */
}
test_coap_client_msg_t;

/**
 *  @brief Client test data structure
 */
typedef struct
{
    const char *desc;                                                           /**< Test description */
    const char *host;                                                           /**< Server host address */
    const char *port;                                                           /**< Server UDP port */
    const char *common_name;                                                    /**< Common name of the server */
    test_coap_client_msg_t test_req;                                           /**< Array of test request message structures */
    test_coap_client_msg_t test_resp;                                          /**< Array of test response message structures */
    size_t num_msg;                                                             /**< Length of the arrays of test message structures */
}
test_coap_client_data_t;

/**
 *  @brief Client structure
 */
typedef struct
{
    int sd;                                                                     /**< Socket descriptor */
    int timer_fd;                                                               /**< Timer file descriptor */
    struct tm timeout;                                                          /**< Timeout value */
    unsigned num_retrans;                                                       /**< Current number of retransmissions */
    int server_sin_len;                                                         /**< Socket structure length */
    char server_host[COAP_CLIENT_HOST_BUF_LEN];                                 /**< String to hold the server host address */
    char server_port[COAP_CLIENT_PORT_BUF_LEN];                                 /**< String to hold the server port number */
}
coap_client_t;

/**
 *  @brief Initialise a client structure
 *
 *  @param[out] client Pointer to a client structure
 *  @param[in] host Pointer to a string containing the host address of the server
 *  @param[in] port Port number of the server
 *
 *  @returns Operation status
 *  @retval 0 Success
 *  @retval <0 Error
 */
int coap_client_create(coap_client_t *client,
                       const char *host,
                       const char *port);


/**
 *  @brief Deinitialise a client structure
 *
 *  @param[in,out] client Pointer to a client structure
 */
void coap_client_destroy(coap_client_t *client);

/**
 *  @brief Send a request to the server and receive the response
 *
 *  @param[in,out] client Pointer to a client structure
 *  @param[in] req Pointer to the request message
 *  @param[out] resp Pointer to the response message
 *
 *  This function sets the message ID and token fields of
 *  the request message overriding any values set by the
 *  calling function.
 *
 *  @returns Operation status
 *  @retval 0 Success
 *  @retval <0 Error
 **/
int coap_client_exchange(coap_client_t *client, coap_msg_t *req, coap_msg_t *resp);

test_result_t test_exchange_func(char* buf,int buf_len,char* returncode);
test_result_t check_resp(test_coap_client_msg_t *test_resp, coap_msg_t *resp);
test_result_t compare_ver_token(coap_msg_t *req, coap_msg_t *resp);
test_result_t exchange(coap_client_t *client, test_coap_client_msg_t *test_req, coap_msg_t *req, coap_msg_t *resp,char* returncode);
test_result_t populate_req(test_coap_client_msg_t *test_req, coap_msg_t *req);
void print_coap_msg(const char *str, coap_msg_t *msg,char* returncode);



#endif

