/*******************************************************************************
 * Copyright (c) 2014, 2015 IBM Corp.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * and Eclipse Distribution License v1.0 which accompany this distribution.
 *
 * The Eclipse Public License is available at
 *    http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 *   http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *    Ian Craggs - initial API and implementation and/or initial documentation
 *******************************************************************************/
 
 /**
  This is a sample program to illustrate the use of the MQTTSN Client library
  on the mbed platform.  The Client class requires two classes which mediate
  access to system interfaces for networking and timing.  As long as these two
  classes provide the required public programming interfaces, it does not matter
  what facilities they use underneath. In this program, they use the mbed
  system libraries.
 
 */
 
 #define WARN printf

#include "EthernetInterface.h"
#include "C12832.h"

#if defined(TARGET_LPC1768)
#warning "Compiling for mbed LPC1768"
#include "LPC1768.h"
#elif defined(TARGET_K64F)
#warning "Compiling for mbed K64F"
#include "K64F.h"
#endif

#define MQTTSNCLIENT_QOS2 1
#include "MQTTSNUDP.h"
#include "MQTTSNClient.h"

int arrivedcount = 0;


void off()
{
    r = g = b = 1.0;    // 1 is off, 0 is full brightness
}

void red()
{
    r = 0.7; g = 1.0; b = 1.0;    // 1 is off, 0 is full brightness
}

void yellow()
{
    r = 0.7; g = 0.7; b = 1.0;    // 1 is off, 0 is full brightness
}

void green()
{
    r = 1.0; g = 0.7; b = 1.0;    // 1 is off, 0 is full brightness
}


void messageArrived(MQTTSN::MessageData& md)
{
    MQTTSN::Message &message = md.message;
    lcd.cls();
    lcd.locate(0,3);
    printf("Message arrived: qos %d, retained %d, dup %d, packetid %d\n", message.qos, message.retained, message.dup, message.id);
    printf("Payload %.*s\n", message.payloadlen, (char*)message.payload);
    ++arrivedcount;
    lcd.puts((char*)message.payload);
}


int main(int argc, char* argv[])
{   
    MQTTSNUDP ipstack = MQTTSNUDP();
    float version = 0.47;
    char* topic = "mbed-sample";
    
    lcd.printf("Version is %f\n", version);
    printf("Version is %f\n", version);
              
    MQTTSN::Client<MQTTSNUDP, Countdown> client = MQTTSN::Client<MQTTSNUDP, Countdown>(ipstack);
    
    EthernetInterface eth;
    eth.init();                          // Use DHCP
    eth.connect();
    
    char* hostname = "9.20.230.78";
    int port = 20000;
    lcd.printf("Connecting to %s:%d\n", hostname, port);
    int rc = ipstack.connect(hostname, port);
    if (rc != 0)
        lcd.printf("rc from TCP connect is %d\n", rc);
    else
        green();
 
    MQTTSNPacket_connectData data = MQTTSNPacket_connectData_initializer;       
    data.clientID.cstring = "mbed-sample";
    data.duration = 60;
    if ((rc = client.connect(data)) != 0)
        lcd.printf("rc from MQTT connect is %d\n", rc);
    
    MQTTSN_topicid topicid;
    topicid.type = MQTTSN_TOPIC_TYPE_NORMAL;
    topicid.data.long_.name = topic;
    topicid.data.long_.len = strlen(topic);
    MQTTSN::QoS grantedQoS;
    if ((rc = client.subscribe(topicid, MQTTSN::QOS1, grantedQoS, messageArrived)) != 0)
        lcd.printf("rc from MQTT subscribe is %d\n", rc);

    MQTTSN::Message message;

    // QoS 0
    char buf[100];
    sprintf(buf, "Hello World!  QoS 0 message from app version %f\n", version);
    message.qos = MQTTSN::QOS0;
    message.retained = false;
    message.dup = false;
    message.payload = (void*)buf;
    message.payloadlen = strlen(buf)+1;
    rc = client.publish(topicid, message);
    while (arrivedcount < 1)
        client.yield(100);
        
    // QoS 1
    sprintf(buf, "Hello World!  QoS 1 message from app version %f\n", version);
    message.qos = MQTTSN::QOS1;
    message.payloadlen = strlen(buf)+1;
    rc = client.publish(topicid, message);
    while (arrivedcount < 2)
        client.yield(100);
        
    // QoS 2
    sprintf(buf, "Hello World!  QoS 2 message from app version %f\n", version);
    message.qos = MQTTSN::QOS2;
    message.payloadlen = strlen(buf)+1;
    rc = client.publish(topicid, message);
    while (arrivedcount < 3)
        client.yield(100);
        
    // n * QoS 2
    for (int i = 1; i <= 10; ++i)
    {
        sprintf(buf, "Hello World!  QoS 2 message number %d from app version %f\n", i, version);
        message.qos = MQTTSN::QOS2;
        message.payloadlen = strlen(buf)+1;
        rc = client.publish(topicid, message);
        while (arrivedcount < i + 3)
            client.yield(100);
    }
    
    if ((rc = client.unsubscribe(topicid)) != 0)
        printf("rc from unsubscribe was %d\n", rc);
    
    if ((rc = client.disconnect()) != 0)
        printf("rc from disconnect was %d\n", rc);
    
    ipstack.disconnect();
    
    lcd.cls();
    lcd.locate(0,3);
    lcd.printf("Version %.2f: finish %d msgs\n", version, arrivedcount);
    printf("Finishing with %d messages received\n", arrivedcount);
    
    return 0;
}
