
#include "mbed.h"
#include <stdio.h>
#include <errno.h>
#include "stm32l475e_iot01_accelero.h"
#include "nvstore.h"

// Block devices
#if COMPONENT_SPIF
#include "SPIFBlockDevice.h"
#endif

#if COMPONENT_DATAFLASH
#include "DataFlashBlockDevice.h"
#endif 

#if COMPONENT_SD
#include "SDBlockDevice.h"
#endif 

#include "HeapBlockDevice.h"
// File systems
#include "LittleFileSystem.h"
#include "FATFileSystem.h"

DigitalOut led1(LED1);
DigitalOut led2(LED2);
DigitalOut led3(LED3);
Ticker toggle_led;

EventQueue queue(32 * EVENTS_EVENT_SIZE);
Thread thread;
int maximum( int, int, int);
//int maximum( int a, int b, int c );

// Physical block device, can be any device that supports the BlockDevice API
/*SPIFBlockDevice bd(
        MBED_CONF_SPIF_DRIVER_SPI_MOSI,
        MBED_CONF_SPIF_DRIVER_SPI_MISO,
        MBED_CONF_SPIF_DRIVER_SPI_CLK,
        MBED_CONF_SPIF_DRIVER_SPI_CS);*/

#define BLOCK_SIZE 512
HeapBlockDevice bd(16384, BLOCK_SIZE);

// File system declaration
LittleFileSystem fs("fs");

// Set up the button to trigger an erase
InterruptIn irq(BUTTON1);
void erase() {
    printf("Initializing the block device... ");
    fflush(stdout);
    int err = bd.init();
    printf("%s\n", (err ? "Fail :(" : "OK"));
    if (err) {
        error("error: %s (%d)\n", strerror(-err), err);
    }

    printf("Erasing the block device... ");
    fflush(stdout);
    err = bd.erase(0, bd.size());
    printf("%s\n", (err ? "Fail :(" : "OK"));
    if (err) {
        error("error: %s (%d)\n", strerror(-err), err);
    }

    printf("Deinitializing the block device... ");
    fflush(stdout);
    err = bd.deinit();
    printf("%s\n", (err ? "Fail :(" : "OK"));
    if (err) {
        error("error: %s (%d)\n", strerror(-err), err);
    }
}

static FILE *file;
volatile int counter = 0;

int rc; 
uint16_t keyX = 1; 
uint32_t valueX;
uint16_t keyY = 2; 
uint32_t valueY;
uint16_t keyZ = 3; 
uint32_t valueZ;

int maximum( int a, int b, int c ) {
   int max = ( a < b ) ? b : a;
   return ( ( max < c ) ? c : max );
}

void readSensors() {
    int16_t pDataXYZ[3] = {0};
    BSP_ACCELERO_AccGetXYZ(pDataXYZ);
    if(pDataXYZ[2] > 900 || pDataXYZ[2] < -900) { //Horizontal State
            fprintf(file, "%d\n", 1);
        } else if(pDataXYZ[1] > 900 || pDataXYZ[1] < -900) { //Long Edge State
            fprintf(file, "%d\n", 2);
        } else if(pDataXYZ[0] > 900 || pDataXYZ[0] < -900) { // Short Edge State
            fprintf(file, "%d\n", 3);
        } else { // all other positions
            fprintf(file, "%d\n", 4);
        }
    fflush(file);
    fflush(stdout);    
}

void readFile() {
    
    int countLed1 = 0;
    int countLed2 = 0;
    int countLed3 = 0;
    
    fflush(stdout);
    fflush(file);
    
    fseek(file, 0, SEEK_SET);
    int ledNumber;
    while(!feof(file)) {
        fscanf(file, "%d", &ledNumber); 
        if(ledNumber == 1){
            countLed1 += 1;    
        } else if(ledNumber == 2){
            countLed2 += 1;    
        } else if(ledNumber == 3){
            countLed3 += 1;    
        }
    }
    int maxLed = maximum(countLed1, countLed2, countLed3);
    printf("MAX LED = %d\n", maxLed);
    
    NVStore &nvstore = NVStore::get_instance();
    if(maxLed == countLed1) {
        led1 = 1;
        led2 = 0;
        led3 = 0;
        valueX += 1;
        nvstore.set(keyX, sizeof(valueX), &valueX);
    } else if (maxLed == countLed2) {
        led1 = 0;
        led2 = 1;
        led3 = 0;
        valueY += 1;
        nvstore.set(keyY, sizeof(valueY), &valueY);
    } else if (maxLed == countLed3) {
        led1 = 0;
        led2 = 0;
        led3 = 1;
        valueZ += 1;
        nvstore.set(keyZ, sizeof(valueZ), &valueZ);
    }
    
    fflush(stdout);
    int err = fclose(file);
    printf("%s\n", (err < 0 ? "Fail :(" : "OK"));
    if (err < 0) {
      error("error: %s (%d)\n", strerror(err), -err);
    }
    err = fs.unmount();
    printf("%s\n", (err < 0 ? "Fail :(" : "OK"));
    if (err < 0) {
        error("error: %s (%d)\n", strerror(-err), err);
    }
      
    printf("Mbed OS filesystem example done!\n"); 
}

void turnOnLed() {
    queue.call(readSensors);
    counter += 1;
    if(counter == 1000) {
        toggle_led.detach();
        queue.call(readFile);
    }
}


// Entry point for the example
int main() {
    thread.start(callback(&queue, &EventQueue::dispatch_forever));
    BSP_ACCELERO_Init();
    
    NVStore &nvstore = NVStore::get_instance();
    rc = nvstore.init();
    uint16_t actual_len_bytes = 0;
    rc = nvstore.get(keyZ, sizeof(valueZ), &valueZ, actual_len_bytes);
    
    if(rc == NVSTORE_NOT_FOUND) {
        valueX = 0;
        valueY = 0;
        valueZ = 0;
        nvstore.set(keyX, sizeof(valueX), &valueX);
        nvstore.set(keyY, sizeof(valueY), &valueY);
        nvstore.set(keyZ, sizeof(valueZ), &valueZ);
    } else {
        nvstore.get(keyX, sizeof(valueX), &valueX, actual_len_bytes);    
        nvstore.get(keyY, sizeof(valueY), &valueY, actual_len_bytes);
    }
    
    printf("Led1 Count: %d -- Led2 Count: %d -- Led3 Count: %d \n", valueX, valueY, valueZ);
    printf("--- Mbed OS filesystem example ---\n");

    // Setup the erase event on button press, use the event queue
    // to avoid running in interrupt context
    irq.fall(mbed_event_queue()->event(erase));

    // Try to mount the filesystem
    printf("Mounting the filesystem... ");
    fflush(stdout);
    int err = fs.mount(&bd);
    printf("%s\n", (err ? "Fail :(" : "OK"));
    if (err) {
        // Reformat if we can't mount the filesystem
        // this should only happen on the first boot
        printf("No filesystem found, formatting... ");
        fflush(stdout);
        err = fs.reformat(&bd);
        printf("%s\n", (err ? "Fail :(" : "OK"));
        if (err) {
            error("error: %s (%d)\n", strerror(-err), err);
        }
    }

    // Open the numbers file
    printf("Opening \"/fs/numbers.txt\"... ");
    fflush(stdout);
    file = fopen("/fs/numbers.txt", "r+");
    printf("%s\n", (!file ? "Fail :(" : "OK"));
    if (!file) {
        // Create the numbers file if it doesn't exist
        printf("No file found, creating a new file... ");
        fflush(stdout);
        file = fopen("/fs/numbers.txt", "w+");
        printf("%s\n", (!file ? "Fail :(" : "OK"));
        if (!file) {
            error("error: %s (%d)\n", strerror(errno), -errno);
        }
    }
    
    toggle_led.attach(&turnOnLed, 0.01);
    

}
