//Hydraulic Control Board
//distributed by Sungwoo Kim
//       2020/12/28


// 유의사항
// 소수 적을때 뒤에 f 꼭 붙이기
// CAN 선은 ground까지 있는 3상 선으로 써야함.
// 전원은 12~24V 인가.

#include "mbed.h"
#include "FastPWM.h"
#include "INIT_HW.h"
#include "function_CAN.h"
#include "SPI_EEP_ENC.h"
#include "I2C_AS5510.h"
#include "setting.h"
#include "function_utilities.h"
#include "stm32f4xx_flash.h"
#include "FlashWriter.h"
#include <string>
#include <iostream>
#include <cmath>

using namespace std;
Timer t;

// dac & check ///////////////////////////////////////////
DigitalOut check(PC_2);
DigitalOut check_2(PC_3);
AnalogOut dac_1(PA_4);
AnalogOut dac_2(PA_5);
AnalogIn adc1(PC_4); //pressure_1
AnalogIn adc2(PB_0); //pressure_2
AnalogIn adc3(PC_1); //current

// PWM ///////////////////////////////////////////
float dtc_v=0.0f;
float dtc_w=0.0f;

// I2C ///////////////////////////////////////////
I2C i2c(PC_9,PA_8); // SDA, SCL (for K22F)
const int i2c_slave_addr1 =  0x56;  // AS5510 address
unsigned int value; // 10bit output of reading sensor AS5510

// SPI ///////////////////////////////////////////
SPI eeprom(PB_15, PB_14, PB_13); // EEPROM //(SPI_MOSI, SPI_MISO, SPI_SCK);
DigitalOut eeprom_cs(PB_12);
SPI enc(PC_12,PC_11,PC_10);
DigitalOut enc_cs(PD_2);
DigitalOut LED(PA_15);

// UART ///////////////////////////////////////////
Serial pc(PA_9,PA_10); //  _ UART

// CAN ///////////////////////////////////////////
CAN can(PB_8, PB_9, 1000000);
CANMessage msg;
void onMsgReceived()
{
    CAN_RX_HANDLER();
}

// Variables ///////////////////////////////////////////
State pos;
State vel;
State Vout;
State torq;
State torq_dot;
State pres_A;
State pres_B;
State cur;
State valve_pos;

State INIT_Vout;
State INIT_Valve_Pos;
State INIT_Pos;
State INIT_torq;

extern int CID_RX_CMD;
extern int CID_RX_REF_POSITION;
extern int CID_RX_REF_OPENLOOP;
extern int CID_RX_REF_PWM;

extern int CID_TX_INFO;
extern int CID_TX_POS_VEL_TORQ;
extern int CID_TX_PWM;
extern int CID_TX_CURRENT;
extern int CID_TX_VOUT;
extern int CID_TX_VALVE_POSITION;
extern int CID_TX_SOMETHING;

float pres_A_new = 0.0f;
float pres_B_new = 0.0f;



float temp_P_GAIN = 0.0f;
float temp_I_GAIN = 0.0f;
int temp_VELOCITY_COMP_GAIN = 0;


/*******************************************************************************
 *  REFERENCE MODE
 ******************************************************************************/
enum _REFERENCE_MODE {
    MODE_REF_NO_ACT = 0,                                //0
    MODE_REF_DIRECT,                                    //1
    MODE_REF_COS_INC,                                   //2
    MODE_REF_LINE_INC,                                  //3
    MODE_REF_SIN_WAVE,                                  //4
    MODE_REF_SQUARE_WAVE,                               //5
};

/*******************************************************************************
 *  CONTROL MODE
 ******************************************************************************/
enum _CONTROL_MODE {
    //control mode
    MODE_NO_ACT = 0,                                    //0
    MODE_VALVE_POSITION_CONTROL,                        //1
    MODE_JOINT_CONTROL,                                 //2

    MODE_VALVE_OPEN_LOOP,                               //3
    MODE_JOINT_ADAPTIVE_BACKSTEPPING,                   //4
    MODE_RL,                                            //5

    MODE_JOINT_POSITION_PRES_CONTROL_PWM,               //6
    MODE_JOINT_POSITION_PRES_CONTROL_VALVE_POSITION,    //7
    MODE_VALVE_POSITION_PRES_CONTROL_LEARNING,          //8

    MODE_TEST_CURRENT_CONTROL,                          //9
    MODE_TEST_PWM_CONTROL,                              //10

    MODE_CURRENT_CONTROL,                               //11
    MODE_JOINT_POSITION_TORQUE_CONTROL_CURRENT,         //12
    MODE_JOINT_POSITION_PRES_CONTROL_CURRENT,           //13
    MODE_VALVE_POSITION_TORQUE_CONTROL_LEARNING,        //14

    //utility
    MODE_TORQUE_SENSOR_NULLING = 20,                    //20
    MODE_VALVE_NULLING_AND_DEADZONE_SETTING,            //21
    MODE_FIND_HOME,                                     //22
    MODE_VALVE_GAIN_SETTING,                            //23
    MODE_PRESSURE_SENSOR_NULLING,                       //24
    MODE_PRESSURE_SENSOR_CALIB,                         //25
    MODE_ROTARY_FRICTION_TUNING,                        //26

    MODE_DDV_POS_VS_PWM_ID = 30,                        //30
    MODE_DDV_DEADZONE_AND_CENTER,                       //31
    MODE_DDV_POS_VS_FLOWRATE,                           //32
    MODE_SYSTEM_ID,                                     //33
    MODE_FREQ_TEST,                                     //34
    MODE_SEND_BUFFER,                                   //35
    MODE_SEND_OVER,                                     //36
    MODE_STEP_TEST,                                     //37
};

void SystemClock_Config(void)
{
    RCC_OscInitTypeDef RCC_OscInitStruct = {0};
    RCC_ClkInitTypeDef RCC_ClkInitStruct = {0};

    /* Configure the main internal regulator output voltage
    */
    __HAL_RCC_PWR_CLK_ENABLE();
    __HAL_PWR_VOLTAGESCALING_CONFIG(PWR_REGULATOR_VOLTAGE_SCALE1);
    /* Initializes the CPU, AHB and APB busses clocks
    */
    RCC_OscInitStruct.OscillatorType = RCC_OSCILLATORTYPE_HSI;
    RCC_OscInitStruct.HSIState = RCC_HSI_ON;
    RCC_OscInitStruct.HSICalibrationValue = RCC_HSICALIBRATION_DEFAULT;
    RCC_OscInitStruct.PLL.PLLState = RCC_PLL_ON;
    RCC_OscInitStruct.PLL.PLLSource = RCC_PLLSOURCE_HSI;
    RCC_OscInitStruct.PLL.PLLM = 8;//8
    RCC_OscInitStruct.PLL.PLLN = 180; //180
    RCC_OscInitStruct.PLL.PLLP = RCC_PLLP_DIV2;
    RCC_OscInitStruct.PLL.PLLQ = 2;
    RCC_OscInitStruct.PLL.PLLR = 2;
    if (HAL_RCC_OscConfig(&RCC_OscInitStruct) != HAL_OK) {
        //Error_Handler();
    }
    /** Activate the Over-Drive mode
    */
    if (HAL_PWREx_EnableOverDrive() != HAL_OK) {
        //Error_Handler();
    }
    /** Initializes the CPU, AHB and APB busses clocks
    */
    RCC_ClkInitStruct.ClockType = RCC_CLOCKTYPE_HCLK|RCC_CLOCKTYPE_SYSCLK
                                  |RCC_CLOCKTYPE_PCLK1|RCC_CLOCKTYPE_PCLK2;
    RCC_ClkInitStruct.SYSCLKSource = RCC_SYSCLKSOURCE_PLLCLK;
    RCC_ClkInitStruct.AHBCLKDivider = RCC_SYSCLK_DIV1;
    RCC_ClkInitStruct.APB1CLKDivider = RCC_HCLK_DIV4;
    RCC_ClkInitStruct.APB2CLKDivider = RCC_HCLK_DIV2;

    if (HAL_RCC_ClockConfig(&RCC_ClkInitStruct, FLASH_LATENCY_5) != HAL_OK) {
        //Error_Handler();
    }
}


int main()
{
    /*********************************
    ***     Initialization
    *********************************/

    HAL_Init();
    SystemClock_Config();

    LED = 0;
    pc.baud(9600);

    // i2c init
//    i2c.frequency(400 * 1000);          // 0.4 mHz
//    wait_ms(2);                         // Power Up wait
//    look_for_hardware_i2c();            // Hardware present
//    init_as5510(i2c_slave_addr1);
//    make_delay();

    // spi init
    eeprom_cs = 1;
    eeprom.format(8,3);
    eeprom.frequency(5000000); //5M
    eeprom_cs = 0;
    make_delay();

    enc_cs = 1;     //sw add
    enc.format(8,0);
    enc.frequency(5000000); //10M
    enc_cs = 0;     //sw add

    make_delay();


    // spi _ enc
    spi_enc_set_init();
    make_delay();

    ////// bno rom
//    spi_eeprom_write(RID_BNO, (int16_t)1);
//    make_delay();
    ////////

    // rom
    ROM_CALL_DATA();
    make_delay();

    // ADC init
    Init_ADC();
    make_delay();

    // Pwm init
    Init_PWM();
    TIM4->CR1 ^= TIM_CR1_UDIS;
    make_delay();

    // CAN
    can.attach(&CAN_RX_HANDLER);
    CAN_ID_INIT();
    make_delay();

    //can.reset();
    can.filter(msg.id, 0xFFFFF000, CANStandard);

    // TMR3 init
    Init_TMR3();
    TIM3->CR1 ^= TIM_CR1_UDIS;
    make_delay();



    //Timer priority
    NVIC_SetPriority(TIM3_IRQn, 2);
    NVIC_SetPriority(TIM4_IRQn, 3);





    //DAC init
    if (SENSING_MODE == 0) {
        dac_1 = TORQUE_VREF / 3.3f;
        dac_2 = 0.0f;
    } else if (SENSING_MODE == 1) {
        if (DIR_VALVE_ENC > 0) {
            dac_1 = PRES_A_VREF / 3.3f;
            dac_2 = PRES_B_VREF / 3.3f;
        } else {
            dac_1 = PRES_B_VREF / 3.3f;
            dac_2 = PRES_A_VREF / 3.3f;
        }
    }
    make_delay();

    for (int i=0; i<50; i++) {
        if(i%2==0)
            ID_index_array[i] = - i * 0.5f;
        else
            ID_index_array[i] =  (i+1) * 0.5f;
    }

    /************************************
    ***     Program is operating!
    *************************************/
    while(1) {

        // UART example
//        if(timer_while==100000) {
//            timer_while = 0;
//            pc.printf("%f\n", value);
//        }
//        timer_while ++;

        //i2c for SW valve
        //if(OPERATING_MODE == 5) {
//            read_field(i2c_slave_addr1);
//            if(DIR_VALVE_ENC < 0) value = 1023 - value;
//        }
    }
}


// Velocity feedforward for SW valve
float DDV_JOINT_POS_FF(float REF_JOINT_VEL)
{
    int i = 0;
    float Ref_Valve_Pos_FF = 0.0f;
    for(i=0; i<VALVE_POS_NUM; i++) {
        if(REF_JOINT_VEL >= min(JOINT_VEL[i],JOINT_VEL[i+1]) && REF_JOINT_VEL <=  max(JOINT_VEL[i],JOINT_VEL[i+1])) {
            if(i==0) {
                if(JOINT_VEL[i+1] == JOINT_VEL[i]) {
                    Ref_Valve_Pos_FF = (float) VALVE_CENTER;
                } else {
                    Ref_Valve_Pos_FF = ((float) 10/(JOINT_VEL[i+1] - JOINT_VEL[i]) * (REF_JOINT_VEL - JOINT_VEL[i])) + (float) VALVE_CENTER;
                }
            } else {
                if(JOINT_VEL[i+1] == JOINT_VEL[i-1]) {
                    Ref_Valve_Pos_FF = (float) VALVE_CENTER;
                } else {
                    Ref_Valve_Pos_FF = ((float) 10*(ID_index_array[i+1] - ID_index_array[i-1])/(JOINT_VEL[i+1] - JOINT_VEL[i-1]) * (REF_JOINT_VEL - JOINT_VEL[i-1])) + (float) VALVE_CENTER + (float) (10*ID_index_array[i-1]);
                }
            }
            break;
        }
    }
    if(REF_JOINT_VEL > max(JOINT_VEL[VALVE_POS_NUM-1], JOINT_VEL[VALVE_POS_NUM-2])) {
        Ref_Valve_Pos_FF = (float) VALVE_MAX_POS;
    } else if(REF_JOINT_VEL < min(JOINT_VEL[VALVE_POS_NUM-1], JOINT_VEL[VALVE_POS_NUM-2])) {
        Ref_Valve_Pos_FF = (float) VALVE_MIN_POS;
    }

    Ref_Valve_Pos_FF = (float) VELOCITY_COMP_GAIN * 0.01f * (float) (Ref_Valve_Pos_FF - (float) VALVE_CENTER);  //VELOCITY_COMP_GAIN : 0~100
    return Ref_Valve_Pos_FF;
}

// Valve feedforward for SW valve
void VALVE_POS_CONTROL(float REF_VALVE_POS)
{
    int i = 0;

    if(REF_VALVE_POS > VALVE_MAX_POS) {
        REF_VALVE_POS = VALVE_MAX_POS;
    } else if(REF_VALVE_POS < VALVE_MIN_POS) {
        REF_VALVE_POS = VALVE_MIN_POS;
    }
    valve_pos_err = (float) (REF_VALVE_POS - value);
    valve_pos_err_diff = valve_pos_err - valve_pos_err_old;
    valve_pos_err_old = valve_pos_err;
    valve_pos_err_sum += valve_pos_err;
    if (valve_pos_err_sum > 1000.0f) valve_pos_err_sum = 1000.0f;
    if (valve_pos_err_sum<-1000.0f) valve_pos_err_sum = -1000.0f;

    VALVE_PWM_RAW_FB = P_GAIN_VALVE_POSITION * valve_pos_err + I_GAIN_VALVE_POSITION * valve_pos_err_sum + D_GAIN_VALVE_POSITION * valve_pos_err_diff;

    for(i=0; i<24; i++) {
        if(REF_VALVE_POS >= min(VALVE_POS_VS_PWM[i],VALVE_POS_VS_PWM[i+1]) && (float) REF_VALVE_POS <=  max(VALVE_POS_VS_PWM[i],VALVE_POS_VS_PWM[i+1])) {
            if(i==0) {
                VALVE_PWM_RAW_FF = (float) 1000.0f / (float) (VALVE_POS_VS_PWM[i+1] - VALVE_POS_VS_PWM[i]) * ((float) REF_VALVE_POS - VALVE_POS_VS_PWM[i]);
            } else {
                VALVE_PWM_RAW_FF = (float) 1000.0f* (float) (ID_index_array[i+1] - ID_index_array[i-1])/(VALVE_POS_VS_PWM[i+1] - VALVE_POS_VS_PWM[i-1]) * ((float) REF_VALVE_POS - VALVE_POS_VS_PWM[i-1]) + 1000.0f * (float) ID_index_array[i-1];
            }
            break;
        }
    }
    Vout.ref = VALVE_PWM_RAW_FF + VALVE_PWM_RAW_FB;
}

// PWM duty vs. voltage output of L6205 in STM board
#define LT_MAX_IDX  57
float LT_PWM_duty[LT_MAX_IDX] = {-100.0f, -80.0f, -60.0f, -50.0f, -40.0f, -35.0f, -30.0f, -25.0f, -20.0f,
                                 -19.0f, -18.0f, -17.0f, -16.0f, -15.0f, -14.0f, -13.0f, -12.0f, -11.0f, -10.0f,
                                 -9.0f, -8.0f, -7.0f, -6.0f, -5.0f, -4.0f, -3.0f, -2.0f, -1.0f, 0.0f,
                                 1.0f, 2.0f, 3.0f, 4.0f, 5.0f, 6.0f, 7.0f, 8.0f, 9.0f, 10.0f,
                                 11.0f, 12.0f, 13.0f, 14.0f, 15.0f, 16.0f, 17.0f, 18.0f, 19.0f, 20.0f,
                                 25.0f, 30.0f, 35.0f, 40.0f, 50.0f, 60.0f, 80.0f, 100.0f
                                };  // duty
float LT_Voltage_Output[LT_MAX_IDX] = {-230.0f, -215.0f, -192.5f, -185.0f, -177.5f, -170.0f, -164.0f, -160.0f, -150.0f,
                                       -150.0f, -145.0f, -145.0f, -145.0f, -135.0f, -135.0f, -135.0f, -127.5f, -127.5f, -115.0f,
                                       -115.0f, -115.0F, -100.0f, -100.0f, -100.0f, -60.0f, -60.0f, -10.0f, -5.0f, 0.0f,
                                       7.5f, 14.0f, 14.0f, 14.0f, 42.5f, 42.5f, 42.5f, 80.0f, 80.0f, 105.0f,
                                       105.0f, 105.0f, 120.0f, 120.0f, 120.0f, 131.0f, 131.0f, 140.0f, 140.0f, 140.0f,
                                       155.0f, 160.0f, 170.0f, 174.0f, 182.0f, 191.0f, 212.0f, 230.0f
                                      }; // mV

float PWM_duty_byLT(float Ref_V)
{
    float PWM_duty = 0.0f;
    if(Ref_V<LT_Voltage_Output[0]) {
        PWM_duty = (Ref_V-LT_Voltage_Output[0])/1.5f+LT_PWM_duty[0];
    } else if (Ref_V>=LT_Voltage_Output[LT_MAX_IDX-1]) {
        PWM_duty = (Ref_V-LT_Voltage_Output[LT_MAX_IDX-1])/1.5f+LT_PWM_duty[LT_MAX_IDX-1];
    } else {
        int idx = 0;
        for(idx=0; idx<LT_MAX_IDX-1; idx++) {
            float ini_x = LT_Voltage_Output[idx];
            float fin_x = LT_Voltage_Output[idx+1];
            float ini_y = LT_PWM_duty[idx];
            float fin_y = LT_PWM_duty[idx+1];
            if(Ref_V>=ini_x && Ref_V<fin_x) {
                PWM_duty = (fin_y-ini_y)/(fin_x-ini_x)*(Ref_V-ini_x) + ini_y;
                break;
            }
        }
    }

    return PWM_duty;
}



/*******************************************************************************
                            TIMER INTERRUPT
*******************************************************************************/

float FREQ_TMR4 = (float)FREQ_20k;
float DT_TMR4 = (float)DT_20k;
long  CNT_TMR4 = 0;
int   TMR4_FREQ_10k = (int)FREQ_10k;
extern "C" void TIM4_IRQHandler(void)
{
    if (TIM4->SR & TIM_SR_UIF ) {
        /*******************************************************
        ***     Sensor Read & Data Handling
        ********************************************************/

        //Encoder
        if (CNT_TMR4 % (int) ((int) FREQ_TMR4/TMR4_FREQ_10k) == 0) {
            ENC_UPDATE();
        }

        ADC1->CR2  |= 0x40000000;
        // Torque Sensing =============================================
        if (SENSING_MODE == 0) {
            float pres_A_new = (((float) ADC1->DR) - 2047.5f);
            double alpha_update_ft = 1.0f / (1.0f + FREQ_TMR4 / (2.0f * 3.14f * 100.0f)); // f_cutoff : 100Hz
            pres_A.sen = (1.0f - alpha_update_ft) * pres_A.sen + alpha_update_ft * pres_A_new;
            torq.sen = -pres_A.sen / TORQUE_SENSOR_PULSE_PER_TORQUE;

            // Pressure Sensing (0~210)bar =============================================
        } else if (SENSING_MODE == 1) {

            if (DIR_VALVE_ENC > 0) {
                pres_A_new = (((float)ADC1->DR) - PRES_A_NULL);
                pres_B_new = (((float)ADC2->DR) - PRES_B_NULL);
            } else {
                pres_A_new = (((float)ADC2->DR) - PRES_A_NULL);
                pres_B_new = (((float)ADC1->DR) - PRES_B_NULL);
            }
            double alpha_update_pres = 1.0f / (1.0f + FREQ_TMR4 / (2.0f * 3.14f * 200.0f)); // f_cutoff : 200Hz
            pres_A.sen = (1.0f - alpha_update_pres) * pres_A.sen + alpha_update_pres * pres_A_new;
            pres_B.sen = (1.0f - alpha_update_pres) * pres_B.sen + alpha_update_pres * pres_B_new;
            CUR_PRES_A_BAR = pres_A.sen / PRES_SENSOR_A_PULSE_PER_BAR;
            CUR_PRES_B_BAR = pres_B.sen / PRES_SENSOR_B_PULSE_PER_BAR;

            if ((OPERATING_MODE & 0x01) == 0) { // Rotary Actuator
                torq.sen = (PISTON_AREA_A * CUR_PRES_A_BAR - PISTON_AREA_B * CUR_PRES_B_BAR) * 0.0001f; // mm^3*bar >> Nm
            } else if ((OPERATING_MODE & 0x01) == 1) { // Linear Actuator
                torq.sen = (PISTON_AREA_A * CUR_PRES_A_BAR - PISTON_AREA_B * CUR_PRES_B_BAR) * 0.1f; // mm^2*bar >> N
            }
        }

        //Current
        //ADC3->CR2  |= 0x40000000;                        // adc _ 12bit
        float alpha_update_cur = 1.0f/(1.0f + FREQ_TMR4/(2.0f*3.14f*500.0f)); // f_cutoff : 500Hz
        float cur_new = ((float)ADC3->DR-2048.0f)*20.0f/4096.0f; // unit : mA
        cur.sen=cur.sen*(1.0f-alpha_update_cur)+cur_new*(alpha_update_cur);

        CNT_TMR4++;
    }
    TIM4->SR = 0x0;  // reset the status register
}


int j =0;
float FREQ_TMR3 = (float)FREQ_5k;
float DT_TMR3 = (float)DT_5k;
int cnt_trans = 0;
double VALVE_POS_RAW_FORCE_FB_LOGGING = 0.0f;
int can_rest =0;

extern "C" void TIM3_IRQHandler(void)
{
    if (TIM3->SR & TIM_SR_UIF ) {


        if (((OPERATING_MODE&0b110)>>1) == 0) {
            K_v = 0.4f; // Moog (LPM >> mA) , 100bar
            mV_PER_mA = 500.0f; // 5000mV/10mA
            mV_PER_pulse = 0.5f; // 5000mV/10000pulse
            mA_PER_pulse = 0.001f; // 10mA/10000pulse
        } else if (((OPERATING_MODE&0b110)>>1) == 1) {
            K_v = 0.5f; // KNR (LPM >> mA) , 100bar
            mV_PER_mA = 166.6666f; // 5000mV/30mA
            mV_PER_pulse = 0.5f; // 5000mV/10000pulse
            mA_PER_pulse = 0.003f; // 30mA/10000pulse
        }

        if(MODE_POS_FT_TRANS == 1) {
            alpha_trans = (float)(1.0f - cos(3.141592f * (float)cnt_trans * DT_TMR3 /3.0f))/2.0f;
            cnt_trans++;
            torq.err_sum = 0;
            if((float)cnt_trans * DT_TMR3 > 3.0f)
                MODE_POS_FT_TRANS = 2;
        } else if(MODE_POS_FT_TRANS == 3) {
            alpha_trans = (float)(1.0f + cos(3.141592f * (float)cnt_trans * DT_TMR3 /3.0f))/2.0f;
            cnt_trans++;
            torq.err_sum = 0;
            if((float) cnt_trans * DT_TMR3 > 3.0f )
                MODE_POS_FT_TRANS = 0;
        } else if(MODE_POS_FT_TRANS == 2) {
            alpha_trans = 1.0f;
            cnt_trans = 0;
        } else {
            alpha_trans = 0.0f;
            cnt_trans = 0;
        }


// Reference Loop ==========================================================
        switch (REFERENCE_MODE) {
            case MODE_REF_NO_ACT: {
                break;
            }

            case MODE_REF_DIRECT: {
                pos.ref = REF_POSITION;
                vel.ref = REF_VELOCITY;
                torq.ref = REF_TORQUE;
                break;
            }
            default:
                break;
        }

        int UTILITY_MODE = 0;
        int CONTROL_MODE = 0;

        if (CONTROL_UTILITY_MODE >= 20 || CONTROL_UTILITY_MODE == 0) {
            UTILITY_MODE = CONTROL_UTILITY_MODE;
            CONTROL_MODE = MODE_NO_ACT;
        } else {
            CONTROL_MODE = CONTROL_UTILITY_MODE;
            UTILITY_MODE = MODE_NO_ACT;
        }



        // UTILITY MODE ------------------------------------------------------------
        switch (UTILITY_MODE) {
            case MODE_NO_ACT: {
                break;
            }

            case MODE_TORQUE_SENSOR_NULLING: {
                // DAC Voltage reference set
                if (TMR3_COUNT_TORQUE_NULL < TMR_FREQ_5k * 2) {
                    CUR_TORQUE_sum += torq.sen;

                    if (TMR3_COUNT_TORQUE_NULL % 10 == 0) {
                        CUR_TORQUE_mean = CUR_TORQUE_sum / 10.0f;
                        CUR_TORQUE_sum = 0;

                        TORQUE_VREF += 0.000003f * (0.0f - CUR_TORQUE_mean);

                        if (TORQUE_VREF > 3.3f) TORQUE_VREF = 3.3f;
                        if (TORQUE_VREF < 0.0f) TORQUE_VREF = 0.0f;
                        dac_1 = TORQUE_VREF / 3.3f;
                    }
                } else {
                    CONTROL_UTILITY_MODE = MODE_NO_ACT;
                    TMR3_COUNT_TORQUE_NULL = 0;
                    CUR_TORQUE_sum = 0;
                    CUR_TORQUE_mean = 0;

                    spi_eeprom_write(RID_TORQUE_SENSOR_VREF, (int16_t) (TORQUE_VREF * 1000.0f));

                    dac_1 = TORQUE_VREF / 3.3f;

                }
                TMR3_COUNT_TORQUE_NULL++;
                break;
            }

            case MODE_FIND_HOME: {
                if (FINDHOME_STAGE == FINDHOME_INIT) {
                    REFERENCE_MODE=MODE_REF_NO_ACT;
                    cnt_findhome = 0;
                    cnt_vel_findhome = 0;
                    pos.ref = pos.sen;
                    vel.ref = 0.0f;
                    FINDHOME_STAGE = FINDHOME_GOTOLIMIT;
                } else if (FINDHOME_STAGE == FINDHOME_GOTOLIMIT) {
                    int cnt_check_enc = (TMR_FREQ_5k/20);
                    if(cnt_findhome%cnt_check_enc == 0) {
                        FINDHOME_POSITION = pos.sen;
                        FINDHOME_VELOCITY = FINDHOME_POSITION - FINDHOME_POSITION_OLD;
                        FINDHOME_POSITION_OLD = FINDHOME_POSITION;
                    }
                    cnt_findhome++;

                    if (abs(FINDHOME_VELOCITY) <= 1) {
                        cnt_vel_findhome = cnt_vel_findhome + 1;
                    } else {
                        cnt_vel_findhome = 0;
                    }

                    if ((cnt_vel_findhome < 3*TMR_FREQ_5k) &&  cnt_findhome < 10*TMR_FREQ_5k) { // wait for 3sec
                        if (HOMEPOS_OFFSET > 0) {
                            pos.ref = pos.ref + 12.0f;
                        } else {
                            pos.ref = pos.ref - 12.0f;
                        }


                        CONTROL_MODE = MODE_JOINT_CONTROL;
                        alpha_trans = 0.0f;

                    } else {
                        ENC_SET((int32_t)((int32_t)HOMEPOS_OFFSET*10));
                        INIT_REF_POS = (int32_t)((int32_t)HOMEPOS_OFFSET*10);
                        REF_POSITION = 0;
                        REF_VELOCITY = 0;
                        FINDHOME_POSITION = 0;
                        FINDHOME_POSITION_OLD = 0;
                        FINDHOME_VELOCITY = 0;
                        cnt_findhome = 0;
                        cnt_vel_findhome = 0;
                        FINDHOME_STAGE = FINDHOME_ZEROPOSE;


                        cnt_findhome = 0;
                        pos.ref = 0.0f;
                        vel.ref = 0.0f;
                        pos.ref_home_pos = 0.0f;
                        vel.ref_home_pos = 0.0f;
                    }
                } else if (FINDHOME_STAGE == FINDHOME_ZEROPOSE) {
                    int T_move = 2*TMR_FREQ_5k;
                    pos.ref = (0.0f - (float)INIT_REF_POS)*0.5f*(1.0f - cos(3.14159f * (float)cnt_findhome / (float)T_move)) + (float)INIT_REF_POS;
                    vel.ref = 0.0f;
                    alpha_trans = 0.0f;

                    pos.err = (pos.ref - pos.sen)/(float)(ENC_PULSE_PER_POSITION); //[mm]
                    vel.err = (0.0f - vel.sen)/(float)(ENC_PULSE_PER_POSITION); //[mm/s]
                    pos.err_sum += pos.err/(float) TMR_FREQ_5k; //[mm]
                    CONTROL_MODE = MODE_JOINT_CONTROL;

                    cnt_findhome++;
                    if (cnt_findhome >= T_move) {
                        cnt_findhome = 0;
                        pos.ref = 0.0f;
                        vel.ref = 0.0f;
                        pos.ref_home_pos = 0.0f;
                        vel.ref_home_pos = 0.0f;
                        FINDHOME_STAGE = FINDHOME_INIT;
                        CONTROL_UTILITY_MODE = MODE_JOINT_CONTROL;
                        REFERENCE_MODE=MODE_REF_DIRECT;
                    }
                }

                break;
            }

            case MODE_PRESSURE_SENSOR_NULLING: {
                // DAC Voltage reference set
                if (TMR3_COUNT_PRES_NULL < TMR_FREQ_5k * 2) {
                    CUR_PRES_A_sum += pres_A.sen;
                    CUR_PRES_B_sum += pres_B.sen;

                    if (TMR3_COUNT_PRES_NULL % 10 == 0) {
                        CUR_PRES_A_mean = CUR_PRES_A_sum / 10.0f;
                        CUR_PRES_B_mean = CUR_PRES_B_sum / 10.0f;
                        CUR_PRES_A_sum = 0;
                        CUR_PRES_B_sum = 0;

                        float VREF_NullingGain = 0.0003f;
                        PRES_A_VREF = PRES_A_VREF + VREF_NullingGain * CUR_PRES_A_mean;
                        PRES_B_VREF = PRES_B_VREF + VREF_NullingGain * CUR_PRES_B_mean;

                        if (PRES_A_VREF > 3.3f) PRES_A_VREF = 3.3f;
                        if (PRES_A_VREF < 0.0f) PRES_A_VREF = 0.0f;
                        if (PRES_B_VREF > 3.3f) PRES_B_VREF = 3.3f;
                        if (PRES_B_VREF < 0.0f) PRES_B_VREF = 0.0f;

                        if (DIR_VALVE_ENC > 0) {
                            dac_1 = PRES_A_VREF / 3.3f;
                            dac_2 = PRES_B_VREF / 3.3f;
                        } else {
                            dac_1 = PRES_B_VREF / 3.3f;
                            dac_2 = PRES_A_VREF / 3.3f;
                        }
                    }
                } else {
                    CONTROL_UTILITY_MODE = MODE_NO_ACT;
                    TMR3_COUNT_PRES_NULL = 0;
                    CUR_PRES_A_sum = 0;
                    CUR_PRES_B_sum = 0;
                    CUR_PRES_A_mean = 0;
                    CUR_PRES_B_mean = 0;

                    spi_eeprom_write(RID_PRES_A_SENSOR_VREF, (int16_t) (PRES_A_VREF * 1000.0f));
                    spi_eeprom_write(RID_PRES_B_SENSOR_VREF, (int16_t) (PRES_B_VREF * 1000.0f));

                    if (DIR_VALVE_ENC > 0) {
                        dac_1 = PRES_A_VREF / 3.3f;
                        dac_2 = PRES_B_VREF / 3.3f;
                    } else {
                        dac_1 = PRES_B_VREF / 3.3f;
                        dac_2 = PRES_A_VREF / 3.3f;
                    }
                }
                TMR3_COUNT_PRES_NULL++;
                break;
            }

            case MODE_DDV_DEADZONE_AND_CENTER: {
                if (FINDDZ_STAGE == FINDDZ_INIT) {
                    cnt_finddz = 0;
                    cnt_vel_finddz = 0;
                    temp_pos_ref = pos.sen;
                    temp_pos_ref_offset = 0.0f;
                    vel.ref = 0.0f;
                    FINDDZ_STAGE = FINDDZ_START1;
                    flag_finddz = 1;
                    FINDDZ_POSITION = pos.sen;
                    FINDDZ_POSITION_OLD = FINDDZ_POSITION;

                    temp_P_GAIN = P_GAIN_JOINT_POSITION;
                    temp_I_GAIN = I_GAIN_JOINT_POSITION;
                    temp_VELOCITY_COMP_GAIN = VELOCITY_COMP_GAIN;

                    P_GAIN_JOINT_POSITION = 1.0f;
                    I_GAIN_JOINT_POSITION = 0.0f;
                    VELOCITY_COMP_GAIN = 0;
                    FLAG_VALVE_DEADZONE = 0;
                    alpha_trans = 0.0f;
                    REFERENCE_MODE=MODE_REF_NO_ACT;

                } else if (FINDDZ_STAGE == FINDDZ_START1) {
                    cnt_finddz++;
                    CONTROL_MODE = MODE_JOINT_CONTROL;
                    FINDDZ_POSITION = pos.sen;
                    FINDDZ_VELOCITY = FINDDZ_POSITION - FINDDZ_POSITION_OLD;
                    FINDDZ_POSITION_OLD = FINDDZ_POSITION;

                    if (temp_P_GAIN > 0.0f) temp_pos_ref_offset = temp_pos_ref_offset - (float)FINDDZ_VELOCITY*temp_P_GAIN;
                    else temp_pos_ref_offset = temp_pos_ref_offset - (float)FINDDZ_VELOCITY*20.0f;

                    pos.ref = temp_pos_ref + temp_pos_ref_offset;

                    if (cnt_finddz > 10000) FINDDZ_STAGE = FINDDZ_START2;


                } else if (FINDDZ_STAGE == FINDDZ_START2) {
                    CONTROL_MODE = MODE_JOINT_CONTROL;
                    FINDDZ_POSITION = pos.sen;
                    FINDDZ_VELOCITY = FINDDZ_POSITION - FINDDZ_POSITION_OLD;
                    FINDDZ_POSITION_OLD = FINDDZ_POSITION;

                    if (flag_finddz > 0) {
                        temp_pos_ref = temp_pos_ref + 1.0f;
                        if (FINDDZ_VELOCITY > 0) {
                            cnt_vel_finddz = cnt_vel_finddz + 1;
                        }
                        if (cnt_vel_finddz > 5) {
                            VALVE_DEADZONE_PLUS = (int)(I_REF_fil/mA_PER_pulse);
//                            temp_VALVE_DEADZONE_PLUS = temp_VALVE_DEADZONE_PLUS + I_REF_fil/mA_PER_pulse;
                            cnt_vel_finddz = 0;
                            flag_finddz = - flag_finddz - 1;
                        }

                    } else if (flag_finddz < 0) {
                        temp_pos_ref = temp_pos_ref - 1.0f;
                        if (FINDDZ_VELOCITY < 0) {
                            cnt_vel_finddz = cnt_vel_finddz + 1;
                        }
                        if (cnt_vel_finddz > 5) {
                            VALVE_DEADZONE_MINUS = (int)(I_REF_fil/mA_PER_pulse);
//                            temp_VALVE_DEADZONE_MINUS = temp_VALVE_DEADZONE_MINUS + I_REF_fil/mA_PER_pulse;
                            cnt_vel_finddz = 0;
                            flag_finddz = - flag_finddz + 1;
                        }
                    }
                    if (abs(flag_finddz) >= 6) {
                        FINDDZ_STAGE = FINDDZ_STOP;
//                        VALVE_DEADZONE_PLUS = (int)(temp_VALVE_DEADZONE_PLUS / (double)flag_finddz * 2.0f);
//                        VALVE_DEADZONE_MINUS = (int)(temp_VALVE_DEADZONE_MINUS / (double)flag_finddz * 2.0f);
                        spi_eeprom_write(RID_VALVE_DEADZONE_PLUS, VALVE_DEADZONE_PLUS);
                        spi_eeprom_write(RID_VALVE_DEADZONE_MINUS, VALVE_DEADZONE_MINUS);
                    }
                    pos.ref = temp_pos_ref + temp_pos_ref_offset;

                } else if (FINDDZ_STAGE == FINDDZ_STOP) {
                    FINDDZ_STAGE = FINDDZ_INIT;
                    CONTROL_UTILITY_MODE = MODE_JOINT_CONTROL;
                    P_GAIN_JOINT_POSITION = temp_P_GAIN;
                    I_GAIN_JOINT_POSITION = temp_I_GAIN;
                    VELOCITY_COMP_GAIN = temp_VELOCITY_COMP_GAIN;
                    flag_finddz = 0;
                    FLAG_VALVE_DEADZONE = 1;
                    REFERENCE_MODE=MODE_REF_DIRECT;
                }
                break;
            }

            case MODE_SYSTEM_ID: {
                freq_sysid_Iref = (double) cnt_sysid * DT_TMR3 * 3.0f;
                valve_pos.ref = 2500.0f * sin(2.0f * 3.14159f * freq_sysid_Iref * (double) cnt_sysid * DT_TMR3);
                CONTROL_MODE = MODE_VALVE_OPEN_LOOP;
                cnt_sysid++;
                if (freq_sysid_Iref >= 300) {
                    cnt_sysid = 0;
                    CONTROL_UTILITY_MODE = MODE_NO_ACT;
                }
                break;
            }

            case MODE_FREQ_TEST: {
                float valve_pos_ref = 2500.0f * sin(2.0f * 3.141592f * freq_test_valve_ref * (float) cnt_freq_test * DT_TMR3);
                if(valve_pos_ref >= 0) {
                    valve_pos.ref = (double)VALVE_CENTER + (double)valve_pos_ref * ((double)VALVE_MAX_POS-(double)VALVE_CENTER)/10000.0f;
                } else {
                    valve_pos.ref = (double)VALVE_CENTER - (double)valve_pos_ref * ((double)VALVE_MIN_POS-(double)VALVE_CENTER)/10000.0f;
                }
                ref_array[cnt_freq_test] = valve_pos_ref;
                if(value>=(float) VALVE_CENTER) {
                    pos_array[cnt_freq_test] = 10000.0f*((double)value - (double)VALVE_CENTER)/((double)VALVE_MAX_POS - (double)VALVE_CENTER);
                } else {
                    pos_array[cnt_freq_test] = -10000.0f*((double)value - (double)VALVE_CENTER)/((double)VALVE_MIN_POS - (double)VALVE_CENTER);
                }

                CONTROL_MODE = MODE_VALVE_POSITION_CONTROL;
                cnt_freq_test++;
                if (freq_test_valve_ref * (float) cnt_freq_test * DT_TMR3 > 2) {
                    buffer_data_size = cnt_freq_test;
                    cnt_freq_test = 0;
                    cnt_send_buffer = 0;
                    freq_test_valve_ref = freq_test_valve_ref * 1.05f;
                    if (freq_test_valve_ref >= 400) {
                        CONTROL_UTILITY_MODE = MODE_NO_ACT;
                        CONTROL_MODE = MODE_NO_ACT;
                        CAN_TX_PWM((int16_t) (1)); //1300
                    }
                    CONTROL_MODE = MODE_NO_ACT;
                    CONTROL_UTILITY_MODE = MODE_SEND_OVER;

                }
                break;
            }

            case MODE_STEP_TEST: {
                float valve_pos_ref = 0.0f;
                if (cnt_step_test < (int) (1.0f * (float) TMR_FREQ_5k)) {
                    valve_pos_ref = 0.0f;
                } else {
                    valve_pos_ref = 10000.0f;
                }
                if(valve_pos_ref >= 0) {
                    valve_pos.ref = (double)VALVE_CENTER + (double)valve_pos_ref * ((double)VALVE_MAX_POS-(double)VALVE_CENTER)/10000.0f;
                } else {
                    valve_pos.ref = (double)VALVE_CENTER - (double)valve_pos_ref * ((double)VALVE_MIN_POS-(double)VALVE_CENTER)/10000.0f;
                }
                ref_array[cnt_step_test] = valve_pos_ref;
                if(value>=(float) VALVE_CENTER) {
                    pos_array[cnt_step_test] = 10000.0f*((double)value - (double)VALVE_CENTER)/((double)VALVE_MAX_POS - (double)VALVE_CENTER);
                } else {
                    pos_array[cnt_step_test] = -10000.0f*((double)value - (double)VALVE_CENTER)/((double)VALVE_MIN_POS - (double)VALVE_CENTER);
                }

                CONTROL_MODE = MODE_VALVE_POSITION_CONTROL;
                cnt_step_test++;
                if (cnt_step_test > (int) (2.0f * (float) TMR_FREQ_5k)) {
                    buffer_data_size = cnt_step_test;
                    cnt_step_test = 0;
                    cnt_send_buffer = 0;
                    CONTROL_UTILITY_MODE = MODE_SEND_OVER;
                    CONTROL_MODE = MODE_NO_ACT;
                }

                break;
            }

            default:
                break;
        }



        // CONTROL MODE ------------------------------------------------------------
        switch (CONTROL_MODE) {
            case MODE_NO_ACT: {
                V_out = 0.0f;
                break;
            }

            case MODE_VALVE_POSITION_CONTROL: {
                if (OPERATING_MODE == 5) { //SW Valve
                    VALVE_POS_CONTROL(valve_pos.ref);
                    V_out = Vout.ref;
                } else if (CURRENT_CONTROL_MODE == 0) { //PWM
                    V_out = valve_pos.ref;
                } else {
                    I_REF = valve_pos.ref * 0.001f;
                }
                break;
            }

            case MODE_JOINT_CONTROL: {

                double torq_ref = 0.0f;
                pos.err = (pos.ref - pos.sen)/(float)(ENC_PULSE_PER_POSITION); //[mm]
                vel.err = (0.0f - vel.sen)/(float)(ENC_PULSE_PER_POSITION); //[mm/s]
                pos.err_sum += pos.err/(float) TMR_FREQ_5k; //[mm]

                //K & D Low Pass Filter
                float alpha_K_D = 1.0f/(1.0f + 5000.0f/(2.0f*3.14f*30.0f)); // f_cutoff : 30Hz
                K_LPF = K_LPF*(1.0f-alpha_K_D)+K_SPRING*(alpha_K_D);
                D_LPF = D_LPF*(1.0f-alpha_K_D)+D_DAMPER*(alpha_K_D);

                torq_ref = torq.ref + K_LPF * pos.err - D_LPF * vel.sen / ENC_PULSE_PER_POSITION; //[N]

                // torque feedback
                torq.err = torq_ref - torq.sen; //[N]
                torq.err_sum += torq.err/(float) TMR_FREQ_5k; //[N]

                if (((OPERATING_MODE&0b110)>>1) == 0 || ((OPERATING_MODE&0b110)>>1) == 1) {

                    double I_REF_POS = 0.0f;
                    double I_REF_FORCE_FB = 0.0f; // I_REF by Force Feedback
                    double I_REF_VC = 0.0f; // I_REF for velocity compensation

                    double temp_vel_pos = 0.0f;
                    double temp_vel_torq = 0.0f;
                    double wn_Pos = 2.0f * PI * 5.0f; // f_cut : 5Hz Position Control

                    if ((OPERATING_MODE & 0x01) == 0) { // Rotary Mode
                        temp_vel_pos = (0.01f * (double) P_GAIN_JOINT_POSITION * wn_Pos * pos.err + 0.01f * (double) I_GAIN_JOINT_POSITION * wn_Pos * pos.err_sum + 0.01f * (double) VELOCITY_COMP_GAIN * vel.ref / ENC_PULSE_PER_POSITION) * PI / 180.0f; // rad/s
                        //                            L when P-gain = 100, f_cut = 10Hz                                 L feedforward velocity
                    } else if ((OPERATING_MODE & 0x01) == 1) {
                        temp_vel_pos = (0.01f * (double) P_GAIN_JOINT_POSITION * wn_Pos * pos.err + 0.01f * (double) I_GAIN_JOINT_POSITION * wn_Pos * pos.err_sum + 0.01f * (double) VELOCITY_COMP_GAIN * vel.ref / ENC_PULSE_PER_POSITION); // mm/s
                        //                            L when P-gain = 100, f_cut = 10Hz                                 L feedforward velocity
                    }
                    if (temp_vel_pos > 0.0f) I_REF_POS = temp_vel_pos * ((double) PISTON_AREA_A * 0.00006f / (K_v * sqrt(2.0f * alpha3 / (alpha3 + 1.0f))));
                    else I_REF_POS = temp_vel_pos * ((double) PISTON_AREA_B * 0.00006f / (K_v * sqrt(2.0f / (alpha3 + 1.0f))));

                    // velocity compensation for torque control
                    if ((OPERATING_MODE & 0x01) == 0) { // Rotary Mode
                        I_REF_FORCE_FB = 0.001f * ((double) P_GAIN_JOINT_TORQUE * torq.err + (double) I_GAIN_JOINT_TORQUE * torq.err_sum);
                        //                temp_vel_torq = (0.01 * (double) VELOCITY_COMP_GAIN * (double) CUR_VELOCITY / (double) ENC_PULSE_PER_POSITION) * PI / 180.0; // rad/s
                        temp_vel_torq = (0.01f * (double) VELOCITY_COMP_GAIN * vel.ref / (double) ENC_PULSE_PER_POSITION) * PI / 180.0f; // rad/s
                        //                                                          L feedforward velocity
                    } else if ((OPERATING_MODE & 0x01) == 1) {
                        I_REF_FORCE_FB = 0.001f * 0.01f*((double) P_GAIN_JOINT_TORQUE * torq.err + (double) I_GAIN_JOINT_TORQUE * torq.err_sum); // Linear Actuators are more sensitive.
                        //                temp_vel_torq = (0.01 * (double) VELOCITY_COMP_GAIN * (double) CUR_VELOCITY / (double) ENC_PULSE_PER_POSITION); // mm/s
                        temp_vel_torq = (0.01f * (double) VELOCITY_COMP_GAIN * vel.ref / (double) ENC_PULSE_PER_POSITION); // mm/s
                        //                                                          L feedforward velocity
                    }
                    if (temp_vel_torq > 0.0f) I_REF_VC = temp_vel_torq * ((double) PISTON_AREA_A * 0.00006f / (K_v * sqrt(2.0f * alpha3 / (alpha3 + 1.0f))));
                    else I_REF_VC = temp_vel_torq * ((double) PISTON_AREA_B * 0.00006f / (K_v * sqrt(2.0f / (alpha3 + 1.0f))));
                    //                                                  L   velocity(rad/s or mm/s) >> I_ref(mA)
                    //            Ref_Joint_FT_dot = (Ref_Joint_FT_Nm - Ref_Joint_FT_Nm_old) / TMR_DT_5k;
                    //            Ref_Joint_FT_Nm_old = Ref_Joint_FT_Nm;

                    I_REF = (1.0f - alpha_trans) * I_REF_POS + alpha_trans * (I_REF_VC + I_REF_FORCE_FB);

                    // Anti-windup for FT
                    if (I_GAIN_JOINT_TORQUE != 0) {
                        double I_MAX = 10.0f; // Maximum Current : 10mV
                        double Ka = 2.0f / ((double) I_GAIN_JOINT_TORQUE * 0.001f);
                        if (I_REF > I_MAX) {
                            double I_rem = I_REF - I_MAX;
                            I_rem = Ka*I_rem;
                            I_REF = I_MAX;
                            torq.err_sum = torq.err_sum - I_rem /(float) TMR_FREQ_5k;
                        } else if (I_REF < -I_MAX) {
                            double I_rem = I_REF - (-I_MAX);
                            I_rem = Ka*I_rem;
                            I_REF = -I_MAX;
                            torq.err_sum = torq.err_sum - I_rem /(float) TMR_FREQ_5k;
                        }
                    }

                } else {
                    float VALVE_POS_RAW_FORCE_FB = 0.0f;
                    float VALVE_POS_RAW_FORCE_FF = 0.0f;
                    float VALVE_POS_RAW = 0.0f;

                    VALVE_POS_RAW_FORCE_FB = alpha_trans*(((float) P_GAIN_JOINT_TORQUE * torq.err + (float) I_GAIN_JOINT_TORQUE * torq.err_sum + (float) D_GAIN_JOINT_TORQUE * (torq.ref_diff - torq_dot.sen)) * 0.01f + DDV_JOINT_POS_FF(vel.sen))+ (1.0f-alpha_trans) * (P_GAIN_JOINT_POSITION * 0.01f * pos.err + DDV_JOINT_POS_FF(vel.ref));

                    VALVE_POS_RAW_FORCE_FF = P_GAIN_JOINT_TORQUE_FF * torq_ref * 0.001f + D_GAIN_JOINT_TORQUE_FF * (torq_ref - torq_ref_past) * 0.0001f;

                    VALVE_POS_RAW = VALVE_POS_RAW_FORCE_FB + VALVE_POS_RAW_FORCE_FF;


                    if (VALVE_POS_RAW >= 0) {
                        valve_pos.ref = VALVE_POS_RAW + VALVE_DEADZONE_PLUS;
                    } else {
                        valve_pos.ref = VALVE_POS_RAW + VALVE_DEADZONE_MINUS;
                    }

                    if(I_GAIN_JOINT_TORQUE != 0) {
                        double Ka = 2.0f / (double) I_GAIN_JOINT_TORQUE * 100.0f;
                        if(valve_pos.ref>VALVE_MAX_POS) {
                            double valve_pos_rem = valve_pos.ref - VALVE_MAX_POS;
                            valve_pos_rem = valve_pos_rem * Ka;
                            valve_pos.ref = VALVE_MAX_POS;
                            torq.err_sum = torq.err_sum - valve_pos_rem/(float) TMR_FREQ_5k;
                        } else if(valve_pos.ref < VALVE_MIN_POS) {
                            double valve_pos_rem = valve_pos.ref - VALVE_MIN_POS;
                            valve_pos_rem = valve_pos_rem * Ka;
                            valve_pos.ref = VALVE_MIN_POS;
                            torq.err_sum = torq.err_sum - valve_pos_rem/(float) TMR_FREQ_5k;
                        }
                    }

                    VALVE_POS_CONTROL(valve_pos.ref);
                    V_out = (float) Vout.ref;
                }
                torq_ref_past = torq_ref;
                break;
            }

            case MODE_VALVE_OPEN_LOOP: {
                V_out = (float) Vout.ref;
                break;
            }

            case MODE_JOINT_ADAPTIVE_BACKSTEPPING: {


                float Va = (1256.6f + Amm * pos.sen/(float)(ENC_PULSE_PER_POSITION)) * 0.000000001f; // 4mm pipe * 100mm + (25mm Cylinder 18mm Rod) * x,      unit : m^3
                float Vb = (1256.6f + Amm  * (79.0f - pos.sen/(float)(ENC_PULSE_PER_POSITION))) * 0.000000001f; // 4mm pipe * 100mm + (25mm Cylinder 18mm Rod) * (79.0mm-x),      unit : m^3

                V_adapt = 1.0f / (1.0f/Va + 1.0f/Vb); //initial 0.0000053f

                //float f3 = -Amm*Amm*beta*0.000001f*0.000001f/V_adapt * vel.sen/(float)(ENC_PULSE_PER_POSITION)*0.001f; // unit : N/s    //xdot=10mm/s일때 -137076
                float f3_hat = -a_hat * vel.sen/(float)(ENC_PULSE_PER_POSITION)*0.001f; // unit : N/s    //xdot=10mm/s일때 -137076

                float g3_prime = 0.0f;
                if (torq.sen > Amm*(Ps-Pt)*0.000001f) {
                    g3_prime = 1.0f;
                } else if (torq.sen < -Amm*(Ps-Pt)*0.000001f) {
                    g3_prime = -1.0f;
                } else {
                    if ((value-VALVE_CENTER) > 0) {
                        g3_prime = sqrt(Ps-Pt-torq.sen/Amm*1000000.0f);
//                        g3_prime = sqrt(Ps-Pt);
                    } else {
                        g3_prime = sqrt(Ps-Pt+torq.sen/Amm*1000000.0f);
//                        g3_prime = sqrt(Ps-Pt);
                    }
                }
                float tau = 0.01f;
                float K_valve = 0.0004f;

                float x_v = 0.0f;   //x_v : -1~1
                if(value>=VALVE_CENTER) {
                    x_v = 1.0f*((double)value - (double)VALVE_CENTER)/((double)VALVE_MAX_POS - (double)VALVE_CENTER);
                } else {
                    x_v = -1.0f*((double)value - (double)VALVE_CENTER)/((double)VALVE_MIN_POS - (double)VALVE_CENTER);
                }
                float f4 = -x_v/tau;
                float g4 = K_valve/tau;

                float torq_ref_dot = torq.ref_diff * 500.0f;

                pos.err = (pos.ref - pos.sen)/(float)(ENC_PULSE_PER_POSITION); //[mm]
                vel.err = (0.0f - vel.sen)/(float)(ENC_PULSE_PER_POSITION); //[mm/s]
                pos.err_sum += pos.err/(float) TMR_FREQ_5k; //[mm]

                torq.err = torq.ref - torq.sen; //[N]
                torq.err_sum += torq.err/(float) TMR_FREQ_5k; //[N]

                float k3 = 2000.0f; //2000  //20000
                float k4 = 10.0f;
                float rho3 = 3.2f;
                float rho4 = 10000000.0f;  //25000000.0f;
                float x_4_des = (-f3_hat + torq_ref_dot - k3*(-torq.err))/(gamma_hat*g3_prime);
                if (x_4_des > 1) x_4_des = 1;
                else if (x_4_des < -1) x_4_des = -1;

                if (x_4_des > 0) {
                    valve_pos.ref = x_4_des * (float)(VALVE_MAX_POS - VALVE_CENTER) + (float) VALVE_CENTER;
                } else {
                    valve_pos.ref = x_4_des * (float)(VALVE_CENTER - VALVE_MIN_POS) + (float) VALVE_CENTER;
                }

                float x_4_des_dot = (x_4_des - x_4_des_old)*(float) TMR_FREQ_5k;
                x_4_des_old = x_4_des;
                V_out = (-f4 + x_4_des_dot - k4*(x_v-x_4_des)- rho3/rho4*gamma_hat*g3_prime*(-torq.err))/g4;

                float rho_a = 0.00001f;
                float a_hat_dot = -rho3/rho_a*vel.sen/(float)(ENC_PULSE_PER_POSITION)*0.001f*(-torq.err);
                a_hat = a_hat + a_hat_dot / (float) TMR_FREQ_5k;

                if(a_hat > -3000000.0f) a_hat = -3000000.0f;
                else if(a_hat < -30000000.0f) a_hat = -30000000.0f;

                break;
            }

            default:
                break;
        }


        if (((OPERATING_MODE&0b110)>>1) == 0 || ((OPERATING_MODE&0b110)>>1) == 1) { //Moog Valve or KNR Valve

            ////////////////////////////////////////////////////////////////////////////
            ////////////////////////////  CURRENT CONTROL //////////////////////////////
            ////////////////////////////////////////////////////////////////////////////
            if (CURRENT_CONTROL_MODE) {
                double alpha_update_Iref = 1.0f / (1.0f + 5000.0f / (2.0f * 3.14f * 300.0f)); // f_cutoff : 500Hz
                I_REF_fil = (1.0f - alpha_update_Iref) * I_REF_fil + alpha_update_Iref*I_REF;

                if (FLAG_VALVE_DEADZONE) {
                    if (I_REF_fil > 0.0f) I_REF_fil_DZ = I_REF_fil + (double)VALVE_DEADZONE_PLUS*mA_PER_pulse; // unit: mA
                    else if (I_REF_fil < 0.0f) I_REF_fil_DZ = I_REF_fil + (double)VALVE_DEADZONE_MINUS*mA_PER_pulse; // unit: mA
                    else I_REF_fil_DZ = I_REF_fil + (double)(VALVE_DEADZONE_PLUS+VALVE_DEADZONE_MINUS)/2.0f*mA_PER_pulse; // unit: mA
                } else {
                    I_REF_fil_DZ = I_REF_fil;
                }
                I_ERR = I_REF_fil_DZ - (double)cur.sen;
                I_ERR_INT = I_ERR_INT + (I_ERR) * 0.0002f;


                // Moog Valve Current Control Gain
                double R_model = 500.0f; // ohm
                double L_model = 1.2f;
                double w0 = 2.0f * 3.14f * 150.0f;
                double KP_I = 0.1f * L_model*w0;
                double KI_I = 0.1f * R_model*w0;

                // KNR Valve Current Control Gain
                if (((OPERATING_MODE & 0b110)>>1) == 1) { // KNR Valve
                    R_model = 163.0f; // ohm
                    L_model = 1.0f;
                    w0 = 2.0f * 3.14f * 80.0f;
                    KP_I = 1.0f * L_model*w0;
                    KI_I = 0.08f * R_model*w0;
                }

                double FF_gain = 1.0f;

                VALVE_PWM_RAW = KP_I * 2.0f * I_ERR + KI_I * 2.0f* I_ERR_INT;
                I_REF_fil_diff = I_REF_fil_DZ - I_REF_fil_old;
                I_REF_fil_old = I_REF_fil_DZ;
//                VALVE_PWM_RAW = VALVE_PWM_RAW + FF_gain * (R_model * I_REF_fil + L_model * I_REF_fil_diff * 5000.0f); // Unit : mV
                VALVE_PWM_RAW = VALVE_PWM_RAW + FF_gain * (R_model * I_REF_fil_DZ); // Unit : mV
                double V_MAX = 12000.0f; // Maximum Voltage : 12V = 12000mV

                double Ka = 3.0f / KP_I;
                if (VALVE_PWM_RAW > V_MAX) {
                    V_rem = VALVE_PWM_RAW - V_MAX;
                    V_rem = Ka*V_rem;
                    VALVE_PWM_RAW = V_MAX;
                    I_ERR_INT = I_ERR_INT - V_rem * 0.0002f;
                } else if (VALVE_PWM_RAW < -V_MAX) {
                    V_rem = VALVE_PWM_RAW - (-V_MAX);
                    V_rem = Ka*V_rem;
                    VALVE_PWM_RAW = -V_MAX;
                    I_ERR_INT = I_ERR_INT - V_rem * 0.0002f;
                }
                Cur_Valve_Open_pulse = cur.sen / mA_PER_pulse;
            } else {
                VALVE_PWM_RAW = I_REF * mV_PER_mA;
                Cur_Valve_Open_pulse = I_REF / mA_PER_pulse;
            }

            ////////////////////////////////////////////////////////////////////////////
            /////////////////  Dead Zone Cancellation & Linearization //////////////////
            ////////////////////////////////////////////////////////////////////////////

            // Output Voltage Linearization
            double CUR_PWM_nonlin = (double)VALVE_PWM_RAW; // Unit : mV
            double CUR_PWM_lin = PWM_duty_byLT(CUR_PWM_nonlin);  // -8000~8000

            // Dead Zone Cancellation (Electrical dead-zone)
            if (CUR_PWM_lin > 0) V_out = (float) (CUR_PWM_lin + 169.0f);
            else if (CUR_PWM_lin < 0) V_out = (float) (CUR_PWM_lin - 174.0f);
            else V_out = (float) (CUR_PWM_lin);
        } else {            //////////////////////////sw valve
            // Output Voltage Linearization & Dead Zone Cancellation (Electrical dead-zone) by SW
            if (V_out > 0 ) V_out = (V_out + 180.0f)/0.8588f;
            else if (V_out < 0) V_out = (V_out - 200.0f)/0.8651f;
            else V_out = 0.0f;
        }

        /*******************************************************
        ***     PWM
        ********************************************************/
        if(DIR_VALVE<0) {
            V_out = -V_out;
        }

        if (V_out >= VALVE_VOLTAGE_LIMIT*1000.0f) {
            V_out = VALVE_VOLTAGE_LIMIT*1000.0f;
        } else if(V_out<=-VALVE_VOLTAGE_LIMIT*1000.0f) {
            V_out = -VALVE_VOLTAGE_LIMIT*1000.0f;
        }
        PWM_out= V_out/(SUPPLY_VOLTAGE*1000.0f);

        // Saturation of output voltage
        if(PWM_out > 1.0f) PWM_out=1.0f;
        else if (PWM_out < -1.0f) PWM_out=-1.0f;

        if (PWM_out>0.0f) {
            dtc_v=0.0f;
            dtc_w=PWM_out;
        } else {
            dtc_v=-PWM_out;
            dtc_w=0.0f;
        }

        //pwm
        TIM4->CCR2 = (PWM_ARR)*(1.0f-dtc_v);
        TIM4->CCR1 = (PWM_ARR)*(1.0f-dtc_w);


        if (TMR2_COUNT_CAN_TX % (int) ((int) TMR_FREQ_5k/CAN_FREQ) == 0) {
            // Position, Velocity, and Torque (ID:1200)
            if (flag_data_request[0] == HIGH) {
                if ((OPERATING_MODE & 0b01) == 0) { // Rotary Actuator
                    if (SENSING_MODE == 0) {
                        CAN_TX_POSITION_FT((int16_t) (pos.sen), (int16_t) (vel.sen/10.0f), (int16_t) (torq.sen*10.0f));
                    } else if (SENSING_MODE == 1) {
                        CAN_TX_POSITION_PRESSURE((int16_t) (pos.sen), (int16_t) (vel.sen/10.0f), (int16_t) ((pres_A.sen)*5.0f), (int16_t) ((pres_B.sen)*5.0f));
                    }
                } else if ((OPERATING_MODE & 0b01) == 1) { // Linear Actuator
                    if (SENSING_MODE == 0) {
                        CAN_TX_POSITION_FT((int16_t) (pos.sen/10.0f), (int16_t) (vel.sen/256.0f), (int16_t) (torq.sen * 10.0f * (float)(TORQUE_SENSOR_PULSE_PER_TORQUE)));
                    } else if (SENSING_MODE == 1) {
                        CAN_TX_POSITION_PRESSURE((int16_t) (pos.sen/10.0f), (int16_t) (vel.sen/256.0f), (int16_t) ((pres_A.sen)*5.0f), (int16_t) ((pres_B.sen)*5.0f));
                    }
                }
            }

            // PWM, ID:1300
            if (flag_data_request[1] == HIGH) {
                CAN_TX_PWM((int16_t) (V_out)); //1300
            }

            // Reference Current, Current Current (ID:1400)
            if (flag_data_request[2] == HIGH) {
                CAN_TX_CURRENT((int16_t) (I_REF_fil_DZ / mA_PER_pulse), (int16_t) (cur.sen / mA_PER_pulse)); // 1400
            }
            
            // Reference Current, Current Current (ID:1700)
            if (flag_data_request[3] == HIGH) {
                CAN_TX_SOMETHING((int16_t) (0), (int16_t) (0), (int16_t) (0), (int16_t) (0)); // 1700
            }

            TMR2_COUNT_CAN_TX = 0;
        }
        TMR2_COUNT_CAN_TX++;

    }
    TIM3->SR = 0x0;  // reset the status register

}