/* mbed Microcontroller Library
 * Copyright (c) 2018 ARM Limited
 * SPDX-License-Identifier: Apache-2.0
 */

#include "mbed.h"
#include <cctype>
#include "Car.h"
#include "AccCar.h"
#include "TextLCD.h"
#include "Road.h"
#include "Intersection.h"
#include "Communication.h"

#include <algorithm>
#include <vector>
#include <queue>
#include <string>
#include <stdlib.h>

Serial pc(USBTX, USBRX);
TextLCD lcd(p15, p16, p17, p18, p19, p20);

#define ROADLENGTH 100
#define STOP 54
#define MIN_SPEED 5
#define MAX_SPEED 15

// Read the max number of services to perform from pc input
int read_int(char* prompt) {
    int maxService = 0;

    pc.printf(prompt);

    char input;
    while(1) {
        input = pc.getc();
        pc.putc(input);

        if( std::isdigit(input) ) {
            maxService = (maxService * 10) + (input-'0');
        } else {
            pc.putc(input);
            break;
        }
    }

    return maxService;
}

// main() runs in its own thread in the OS
int main()
{

    pc.printf("\nStarting simulation\n");
    // ------------------------------------------------------------------------------
    // The following three variables are used for timing statistics, do not modify them
    Timer stopwatch;    // A timer to keep track of how long the updates take, for statistics purposes
    int numberCycles = 0;
    int totalUpdateTime = 0;
    // ------------------------------------------------------------------------------

    Intersection intersection;

    // Initialize Communication
    
    char buf1[30] = "Rahman/Position/2";
    char buf2[30] = "Rahman/Control/2";
    char buf3[50] = "Rahman/Sync/Receive/2";
    char buf4[50] = "Rahman/Sync/Send/2";
    Communication* c = Communication::getInstance(buf1, buf2, buf3, buf4);
    
    // Initialize 5 AccCars and the Road
    
    Road road1(c);
    intersection.road1 = &road1;
    road1.intersection = &intersection;
    
    
    AccCar car11(1, &road1, 0x01, c);
    AccCar car12(2, &road1, 0x02, c);
    AccCar car13(3, &road1, 0x04, c);
    AccCar car14(4, &road1, 0x08, c);
    AccCar car15(5, &road1, 0x10, c);

    std::vector<AccCar*> q1;
    q1.push_back(&car15);
    q1.push_back(&car14);
    q1.push_back(&car13);
    q1.push_back(&car12);
    q1.push_back(&car11);

    for (int i = 0; i < 4; i++) {
        q1[i]->set_forward_car(q1[i+1]);
    }

    AccCar *lastCar1 = q1.front();

    stopwatch.start();
    
    pc.printf("Dispatching communication thread.\r\n");
    
    c->reset();
    
    int interval = MAX_SPEED - MIN_SPEED + 1;
    car11.reset(rand() % interval + MIN_SPEED); // set random speed [5, 15]
    car12.reset(rand() % interval + MIN_SPEED);
    car13.reset(rand() % interval + MIN_SPEED);
    car14.reset(rand() % interval + MIN_SPEED);
    car15.reset(rand() % interval + MIN_SPEED);

//    c->reset();
    stopwatch.reset();

    int waitTime1 = 0;

    do {
        int enterRoad1 = -1;
        int enterCross1 = -1;

        if (numberCycles == 0) {
            // first car enters unconditionally
            AccCar *car1 = q1.back();
            enterRoad1 = car1->id;
            road1.add_acc_car(car1, car1->id);
            q1.pop_back();
            waitTime1 = rand() % 4;
        }
        else {
          if (q1.size() > 0) {
            if (waitTime1 == 0) {
                AccCar *car1 = q1.back();
                if (car1->forward_car->position - 2 >= car1->speed) {
                    road1.add_acc_car(car1, car1->id);
                    enterRoad1 = car1->id;
                    q1.pop_back();
                    waitTime1 = rand() % 4;
                }
            } else {
                waitTime1 = waitTime1 - 1;
            }
          }
        }

        road1.let_cars_update();
        road1.wait_for_car_update();
        // ------------------------------------------------------------------
        // Timing statistics logic, do not modify
        totalUpdateTime += stopwatch.read_ms();
        numberCycles++;
        stopwatch.reset();
        // ------------------------------------------------------------------

        if (car11.position == STOP) {
            enterCross1 = 1;
        } else if (car12.position == STOP) {
            enterCross1 = 2;
        } else if (car13.position == STOP) {
            enterCross1 = 3;
        } else if (car14.position == STOP) {
            enterCross1 = 4;
        } else if (car15.position == STOP) {
            enterCross1 = 5;
        }
        lcd.cls();
        if (enterRoad1 == -1 && enterCross1 == -1) {
            lcd.printf("x, x\n");
        } else if (enterRoad1 == -1) {
            lcd.printf("x, %d\n", enterCross1);
        } else if (enterCross1 == -1) {
            lcd.printf("%d, x\n", enterRoad1);
        } else {
            lcd.printf("%d, %d\n", enterRoad1, enterCross1);
        }

        pc.printf("Car 1 on road 1: position %d, speed %d\r\n", car11.position, car11.speed);
        pc.printf("Car 2 on road 1: position %d, speed %d\r\n", car12.position, car12.speed);
        pc.printf("Car 3 on road 1: position %d, speed %d\r\n", car13.position, car13.speed);
        pc.printf("Car 4 on road 1: position %d, speed %d\r\n", car14.position, car14.speed);
        pc.printf("Car 5 on road 1: position %d, speed %d\r\n", car15.position, car15.speed);

    } while (lastCar1->position <= ROADLENGTH);
    car11.stop();
    car12.stop();
    car13.stop();
    car14.stop();
    car15.stop();
    c->stop();

            // ----------------------------------------------------------------------
    // Timing statistics printout, do not modify
    pc.printf("Average update cycle took: %fms \r\n", (totalUpdateTime*1.0)/(numberCycles*1.0));
    totalUpdateTime = 0;
    numberCycles = 0;
    // ----------------------------------------------------------------------
}
