/**
@file N5110.cpp

@brief Member functions implementations

*/
#include "mbed.h"
#include "N5110.h"


N5110::N5110(PinName pwrPin, PinName scePin, PinName rstPin, PinName dcPin, PinName mosiPin, PinName sclkPin, PinName ledPin)
{
    
    spi = new SPI(mosiPin,NC,sclkPin); // create new SPI instance and initialise
    spi->format(LCD_SPI_BITS, LCD_SPI_MODE);
    spi->frequency(LCD_FREQ);

    // set up pins as required
    led = new PwmOut(ledPin);
    //pwr = new DigitalOut(pwrPin);
    sce = new DigitalOut(scePin);
    rst = new DigitalOut(rstPin);
    dc = new DigitalOut(dcPin);
    W = 83;
    H = 47;

}

// initialise function - powers up and sends the initialisation commands
void N5110::init()
{
    turnOn();     // power up
    wait_ms(10);  // small delay seems to prevent spurious pixels during mbed reset
    reset();      // reset LCD - must be done within 100 ms

    // function set - extended
    sendCommand(0x20 | CMD_FS_ACTIVE_MODE | CMD_FS_HORIZONTAL_MODE | CMD_FS_EXTENDED_MODE);
    // Don't completely understand these parameters - they seem to work as they are
    // Consult the datasheet if you need to change them
    sendCommand(CMD_VOP_7V38);    // operating voltage - these values are from Chris Yan's Library
    sendCommand(CMD_TC_TEMP_2);   // temperature control
    sendCommand(CMD_BI_MUX_48);   // bias

    // function set - basic
    sendCommand(0x20 | CMD_FS_ACTIVE_MODE | CMD_FS_HORIZONTAL_MODE | CMD_FS_BASIC_MODE);
    normalMode();  // normal video mode by default
    sendCommand(CMD_DC_NORMAL_MODE);  // black on white

    // RAM is undefined at power-up so clear
    clearRAM();

}
 
// sets normal video mode (black on white) 
void N5110::normalMode() {
     sendCommand(CMD_DC_NORMAL_MODE);  
   
}

// sets normal video mode (white on black) 
void N5110::inverseMode() {
    sendCommand(CMD_DC_INVERT_VIDEO); 
}

// function to power up the LCD and backlight
void N5110::turnOn()
{
    // set brightness of LED - 0.0 to 1.0 - default is 50%
    setBrightness(0.5);
    //led->write(0);  // apply power
}

// function to power down LCD
void N5110::turnOff()
{
    setBrightness(1.0);  // turn backlight off
    clearRAM();   // clear RAM to ensure specified current consumption
    // send command to ensure we are in basic model
    sendCommand(0x20 | CMD_FS_ACTIVE_MODE | CMD_FS_HORIZONTAL_MODE | CMD_FS_BASIC_MODE);
    // clear the display
    sendCommand(CMD_DC_CLEAR_DISPLAY);
    // enter the extended mode and power down
    sendCommand(0x20 | CMD_FS_POWER_DOWN_MODE | CMD_FS_HORIZONTAL_MODE | CMD_FS_EXTENDED_MODE);
    // small delay and then turn off the power pin
    wait_ms(10);
   // pwr->write(1);

}

// function to change LED backlight brightness
void N5110::setBrightness(float brightness)
{
    // check whether brightness is within range
    if (brightness < 0.0)
        brightness = 0.0;
    if (brightness > 1.0)
        brightness = 1.0;
    // set PWM duty cycle
    led->write(brightness);
}


// pulse the active low reset line
void N5110::reset()
{
    rst->write(0);  // reset the LCD
    rst->write(1);
}



// send a command to the display
void N5110::sendCommand(unsigned char command)
{
    dc->write(0);  // set DC low for command
    //sce->write(0); // set CE low to begin frame
    spi->write(command);  // send command
    dc->write(1);  // turn back to data by default
    //sce->write(1); // set CE high to end frame (expected for transmission of single byte)

}

// send data to the display at the current XY address
// dc is set to 1 (i.e. data) after sending a command and so should
// be the default mode.
void N5110::sendData(unsigned char data)
{
    //sce->write(0);   // set CE low to begin frame
    spi->write(data);
    //sce->write(1);  // set CE high to end frame (expected for transmission of single byte)
}

// this function writes 0 to the 504 bytes to clear the RAM
void N5110::clearRAM()
{
    int i;
    //sce->write(0);  //set CE low to begin frame
    for(i = 0; i < 504; i++) { // 48 x 84 bits = 504 bytes
        spi->write(0x00);  // send 0's
    }
    //sce->write(1); // set CE high to end frame

}

// function to set the XY address in RAM for subsequenct data write 
void N5110::setXYAddress(int x, int y)
{
    // check whether address is in range
    if (x > 83)
        x=83;
    if (y > 5)
        y=5;
    if (x < 0)
        x=0;
    if (y < 0)
        y=0;

    sendCommand(0x80 | x);  // send addresses to display with relevant mask
    sendCommand(0x40 | y);
}

// These functions are used to set, clear and get the value of pixels in the display
// Pixels are addressed in the range of 0 to 47 (y) and 0 to 83 (x).  The refresh()
// function must be called after set and clear in order to update the display
void N5110::setPixel(int x, int y)
{
    // calculate bank and shift 1 to required position in the data byte
    buffer[x][y/8] |= (1 << y%8);
}

void N5110::clearPixel(int x, int y)
{
    // calculate bank and shift 1 to required position (using bit clear)
    buffer[x][y/8] &= ~(1 << y%8);
}

void N5110::drawHline(int x, int y, int l)
{
	for(int i=0; i<l; i++)
	{
		int ps = x+i;
		if ((ps < 84 && ps >= 0) && (y<48 && y>=0)) {
			buffer[ps][y/8] |= (1 << y%8);
			//refresh();
		}

	}
	refresh();
}

void N5110::drawVline(int x, int y, int l)
{
	for(int i=0; i<l; i++)
	{
		int ps = y+i;
		if ((ps < 48 && ps >= 0) && (x<84 && x>=0)){
			buffer[x][ps/8] |= (1 << ps%8);
			//refresh();
		}
	}
	refresh();
}

void N5110::drawRectangle(int x, int y, int w, int h)
{
	drawHline(x,y,w);
	drawVline(x+w,y,h);
	drawHline(x,y+h,w);
	drawVline(x,y,h);
}

void N5110::drawGrid(int stepx, int stepy)
{
	for (int y=0; y<47; y += stepy){
		for(int x=0; x<84; x++){
			setPixel(x,y);
		 	}
		  refresh();
	}
	for (int x=0; x<84; x += stepx){
		for(int y=0; y<47; y++){
			setPixel(x,y);
		 }
		 refresh();
	}
}

// draw line between two points using C implementation of Bresenham's line algorithm
void N5110::drawLine(int x0, int y0, int x1, int y1) {

  int dx = abs(x1-x0), sx = x0<x1 ? 1 : -1;
  int dy = abs(y1-y0), sy = y0<y1 ? 1 : -1;
  int err = (dx>dy ? dx : -dy)/2, e2;

  for(;;){
	if ((x0 < 84 && x0 >= 0) && (y0<47 && y0>=0)) {
			buffer[x0][y0/8] |= (1 << y0%8);
		}
    if (x0==x1 && y0==y1) break;
    e2 = err;
    if (e2 >-dx) { err -= dy; x0 += sx; }
    if (e2 < dy) { err += dx; y0 += sy; }
  }
  refresh();
}


void N5110::drawLineAngle(int x0, int y0, int l, float angle) {
	float radian = 0-angle * (M_PI/180);
	int x1 = x0 + (l*cos(radian));
	int y1 = y0 + (l*sin(radian));
	drawLine(x0,y0,x1,y1);
}


void N5110::drawCircle(int x, int y, int radius, int divisions) {

	float angleIncrement = M_PI * 2 / divisions;
	double tempAngle = 0.0;

	for (int i = 0; i < divisions; ++i) {
		double px = cos(tempAngle) * radius + x;
		double py = sin(tempAngle) * radius + y;
		if ((px < 84 && px >= 0) && (py<47 && py>=0)) {
				buffer[(int)px][(int)py/8] |= (1 << (int)py%8);
			}
		tempAngle += angleIncrement;
	    }
	refresh();
}

unsigned char N5110::getPixel(int x, int y)
{
    // return relevant bank and mask required bit
    return buffer[x][y/8] & (1 << y%8);

}

// function to refresh the display
void N5110::refresh()
{
    int i,j;
  //  sce->write(0);  //set CE low to begin frame

    for(j = 0; j < 6; j++) {  // be careful to use correct order (j,i) for horizontal addressing
        for(i = 0; i < 84; i++) {
            spi->write(buffer[i][j]);  // send buffer
        }
    }
  //  sce->write(1); // set CE high to end frame

}

// fills the buffer with random bytes.  Can be used to test the display.
// The rand() function isn't seeded so it probably creates the same pattern everytime
void N5110::randomiseBuffer()
{
    int i,j;
    for(j = 0; j < 6; j++) {  // be careful to use correct order (j,i) for horizontal addressing
        for(i = 0; i < 84; i++) {
            buffer[i][j] = rand()%256;  // generate random byte
        }
    }

}

// function to print 5x7 font
void N5110::printChar(char c)
{
    int i;
    // loop through 5 columns
    for (i = 0; i < 5 ; i++ ) {
        sendData((font5x7[(c - 32)*5 + i]));
        // array is offset by 32 relative to ASCII, each character is 5 pixels wide
        // the X address is automatically incremented after each data write
    }
    sendData(0); // send an empty byte to introduce space between characters

}

// function to print 6x7 negative font
void N5110::printNegChar(char c)
{
    int i;
    // loop through 5 columns
    for (i = 0; i < 6 ; i++ ) {
        sendData(~(font6x7[(c - 32)*6 + i]));
        // array is offset by 32 relative to ASCII, each character is 5 pixels wide
        // the X address is automatically incremented after each data write
    }
    sendData(0); // send an empty byte to introduce space between characters

}

// function to print string at specified position
void N5110::printString(const char * str,int x,int y)
{
    int n = 0 ; // counter for number of characters in string
    // loop through string and print character
    while(*str) {

        setXYAddress(x+6*n,y);  // leave 1 pixel (6 = 5 + 1) between each character
        printChar(*str);   // print the char - can probably so *str++ and remove next line
        str++;  // go to next character in string
        n++;    // increment index
    }

}

// function to print string at specified position
void N5110::printNegString(const char * str,int x,int y)
{
    int n = 0 ; // counter for number of characters in string
    // loop through string and print character
    while(*str) {

        setXYAddress(x+6*n,y);  // leave 1 pixel (6 = 5 + 1) between each character
        printNegChar(*str);   // print the char - can probably so *str++ and remove next line
        str++;  // go to next character in string
        n++;    // increment index
    }

}

// function to clear the screen
void N5110::clear()
{
    clearBuffer();  // clear the buffer then call the refresh function
    refresh();
}

// function to clear the buffer
void N5110::clearBuffer()
{
    int i,j;
    for (i=0; i<84; i++) {  // loop through the banks and set the buffer to 0
        for (j=0; j<6; j++) {
            buffer[i][j]=0;
        }
    }
}
