// S16_Blue_ToF - Blue button and Time-of-Flight Sensor Service over GATT
 
#include "bricks/bricks.h"
#include "shields/shields.h"

   typedef DisplayShield Dish;
   
   static Dish *pShield;
   Blinker blink;

//==============================================================================
// GATT Database Setup
//==============================================================================
   
// Detection Service

   Service detection(0xA010,"Detection"); // Detection Service
   Characteristic<Bool>        chrPresence(detection,0xA011, "n", "Presence"); 

// Debug Service

   Service debug(0xA030,"Debug");      // Debug Service
   Characteristic<Bool>        chrTest      (debug, 0xA031, "w", "Test"); 

   static void cbWritten(Blob &o)            // handle updates
   {
      Bool value;  
      
      if (updated(o,chrTest))          // has chrTest been updated?
      {
         get(o,chrTest,value);         // get value of chrTest
         set(o,chrPresence,value);     // and store this value to chrPresence 
      }
   }

   void services(Blob &o)
   {
      enroll(o,detection);             // enroll detection service
      enroll(o,debug);                 // enroll debug service
       
      onWritten(o,cbWritten);          // setup 'data written' callback
   }
   
//==============================================================================
// Button Functionality
//==============================================================================

   typedef enum state { OFF, ON, IDLE } State;
   
   InterruptIn button(USER_BUTTON);    // declare blue user button
   State state = IDLE; 
   
   static void cbRise(void)
   {
      Blob o;                          // declare a blob (BLE OBject)
      Bool value = 0;  

      if (o.hasInitialized())
      {
         blink.off();;
         state = OFF;
         set(o,chrPresence,value);     // and store this value to chrPresence 
      }
   }

   static void cbFall(void)
   {
      Blob o;                          // declare a blob (BLE OBject)
      Bool value = 1;  

      if (o.hasInitialized())
      {
         blink.on();
         state = ON;
         set(o,chrPresence,value);     // and store this value to chrPresence 
      }
   }

   void setupButton()                  // enroll button functionality
   {
      button.rise(&cbRise);            // interrupt callback setup
      button.fall(&cbFall);            // interrupt callback setup
   }

//==============================================================================
// Setup BLE Functionality
//==============================================================================

   void cbError(Blob &o)               // Error Reporting Callback
   {
       blink.error();                  // 'error' blink sequence
   }    

   void cbConnect(Blob &o)             // Connection Callback
   {
      blink.connected();               // 'error' blink sequence
   }

   void cbDisconnect(Blob &o)          // Disconnection Callback
   {
      advertise(o);                    // start advertising on client disconnect
      blink.advertise();               // 'advertise' blink sequence
   }

   void cbInitComplete(Blob &o)        // Immediately After Initializing BLE 
   {
      services(o);                     // enroll all services & setup callbacks

      onConnect(o,cbConnect);          // setup connection callback
      onDisconnect(o,cbDisconnect);    // setup disconnection callback
   
      device(o,"S16#2.0 Blue TOF");
      name(o,"Blue TOF");      
         
      advertise(o,"C:ng",100);         // start advertising @ 100 msec interval
      blink.advertise();               // 'advertise' blink sequence
   }

//==============================================================================
// Simple Loop - Cannot Change Mode
//==============================================================================

   void distanceMeasurement(Dish &d)
   {
      uint32_t dist;

      pShield->pBoard->sensor_top->GetDistance(&dist);

      if (dist == 0xFF)
         dist = 0xFFFFFFFF;

      d.data.range_mm = dist;
   }
   
   
   void refreshDisplay(Dish &d, int msec)
   {
      Timer timer;
   
      timer.start();
      for (int i=0; i < msec; i = timer.read_ms())
         d.refresh();                  // refresh display
      timer.stop();
   }
   
   
   void loop(Blob &o, Dish &d)
   {
      static const char *off = "          ";
      Bool state = false;

      for (;;)
      {
         distanceMeasurement(d);
         if (d.data.range_mm != 0xFFFFFFFF)
         {  if (state == false)
            {  blink.on();             // switch LED on
               state = true;
               set(o,chrPresence,state);
            }
         }
         else
         {  if (state == true)
            {  if (isConnected(o))
                  blink.connected(off);   // switch LED off
               else
                  blink.advertise(off);   // switch LED off
               state = false;
               set(o,chrPresence,state);
            }
         }

         if (d.red())                  // red slider switched to 'RANGE'?
            refreshDisplay(d,250);
         sleep(o);                     // 'sleep(o,10)' causes connect issues
      }
   }    

//==============================================================================
// Main Program
//==============================================================================

   int main(void)
   {
      Blob o;
      verbose(o);                      // enable all trace messages
      trace(o,1,"\nS16_Blue_ToF\n");
      blink.idle();
      
      pShield = new Dish("Init");      // init shield
      pShield->setup(InterruptTOF);    // setup fot ToF measurement only
         
      init(o,cbInitComplete);          // init BLE system
      setupButton();                   // enroll button functionality
            
      loop(o,*pShield);                // run measurement loop
   }

