// collection.h - used to setup a collection of characteristics in order to
//                setup a service.
//
// Description:
//    Collections are used to setup a service comprising a collection of
//    characteristics. The setup sequence starts with declaring a collection
//    (this constructs an empty collection) while passing the collection in-
//    stance repeatedly to a constructor of a characteristic, which auto-adds
//    the new constructed characteristics to the collection. Finally, when the
//    collection instance is passed to a service constructor, the service con-
//    structor is able to access all characteristics which have been collected
//    previously.  
//
// Example 1: A protocol might be declared as a class as follows
//
//    Collection col;                            // collection used during setup
//
//    Characteristic<ObjectId>   id(col,0x2AC3,"rw","ID");
//    Characteristic<ObjectName> name(col,0x2ABE,"rw","Name");
//    Characteristic<Digital>    presence(col,0x2A56,"r","Presence");
//
//    Service presenceDetection(col,0xA001);     // instantiate service
//
//    onSetup(Blob &blue)
//    {
//       blue.service(presenceDetection);        // add service    
//    }
//
// Example 2: service definition by means of a service definition class
//
//   class PresenceDetector
//   {
//      public:
//         Collection col;                       // collection used during setup 
//
//         Characteristic<ObjectId> id;          // ID of presence detector
//         Characteristic<ObjectName> name;      // name of presence detector
//         Characteristic<Digital> presence;     // digital presence value
//         Characteristic<DateTime> timestamp;   // last detection change's time
//         Characteristic<ObjectName> layout;    // name of model railway layout
//
//         Service presenceDetection;            // the service
//
//      public:
//         PresenceDetector(Blob &blue, cost UUID uuid) :
//            list;                              // init service list
//            id(list,0x2AC3,"rw","ID"),         // instantiate characteristic
//            name(list,0x2ABE,"rw","Name"),     // instantiate characteristic
//            presence(list,0x2A56,"r","Presence"),// instantiate characteristic
//            timestamp(list,0x2A08,"r","Timestamp"),// instantiate characteristic
//            layout(list,0x2ABE,"rw","Layout"), // instantiate characteristic
//            presenceDetection(list,uuid)       // instantiate service
//         {
//            blue.service(presenceDetection);   // add service    
//         }
//   };       
//
#ifndef _COLLECTION_H_
#define _COLLECTION_H_

#include "ble/BLE.h"
#include "ble/Gap.h"

   class Collection
   {
      private:
         typedef GattCharacteristic *GattCharPtr;
         
      public:
         GattCharacteristic **plist;
         int count; 
         
         Collection()                  // construct empty collection
         {
             plist = 0;  count = 0;    // init properties
         } 
         
         void add(GattCharPtr p)
         {
             GattCharPtr *pnew = new GattCharPtr[count+1];
             pnew[count] = p;          // add new characteristic

             if (count > 0)
             {  for (int i=0; i < count; i++)
                   pnew[i] = plist[i];
                delete plist;  
             }
             plist = pnew;             // update list pointer
             count++;
         }
   };

#endif // _COLLECTION_H_
