// enroll.h - enroll a service
//
// Synopsis:
//
//    void enroll(Blob &o, GattService &gservice);
//    void enroll(Blob &o, uint16_t bcid = 0xFFFF);
//
//    void enroll(Blob &o, Service &service, uint16_t bcid = 0xFFFF);
//
// Arguments:
//
//    o:        Blob object (to avoid name clashes)
//    gservice: The GattService to be enrolled
//    service:  The Service to be enrolled
//    bcid:     Broadcast ID (optional); if not provided the default value
//              0xFFFF will be used
//
// Description
//
//    There are three ways to call enrollment.
// 
//    In the first case the service has been setup via a GattService class.
//    The service is enrolled by registering the service @ GAP. On advertising
//    the provided broadcast ID is used (otherwise default ID 0xFFFF).
//
//    The second case is based on a Service class object which has been setup
//    by having added a set of characteristics to the internal collection. Upon
//    enrollment a GattService instance will be created internally and enroll-
//    ment of this GattService will be performed according to the first case.  
//
//    The third way is to enroll only the service id. This calling syntax is
//    used, if a Gatt servive is pre-enrolled without enrolling the service ID.
//
// Example 1: enrollment of GattService
//
//    enroll(o,gservice);
//    enroll(o,0xFFFF);
// 
// See also: SERVICE
//
#ifndef _ENROLL_H_
#define _ENROLL_H_

#include "bricks/blob.h"
#include "bricks/service.h"


   inline void enroll(Blob &o, uint16_t bcid = 0xFFFF)
   {
      static uint16_t list[1];
      
         // Custom UUID, FFFF is reserved for development
         // Used for UUID's broadcast in advertising packet 

      list[0] = bcid;            // set broadcast ID

//    o.pble->gap().accumulateAdvertisingPayload(
      o.gap().accumulateAdvertisingPayload(
         GapAdvertisingData::COMPLETE_LIST_16BIT_SERVICE_IDS,
         (uint8_t *)list, sizeof(list));
   }


   inline void enroll(Blob &o, GattService &gservice, uint16_t bcid = 0xFFFF)
   {
//    o.pble->addService(gservice);    // add service to GATT attributes
      o.addService(gservice);          // add service to GATT attributes
      enroll(o,bcid);                  // enroll GattService (without BCID)
   }


   inline void enroll(Blob &o, Service &service, uint16_t bcid = 0)
   {
      Collection &collection = service.collection;
      GattService gservice(service.uuid, collection.plist, collection.count);

      if (bcid == 0)
         bcid = service.uuid;

      enroll(o,gservice,bcid);         // enroll GattService (with BCID)
   }

#endif // _ENROLL_H_