// advertise.h - begin peripheral role and start advertising
//
// Synopsis:
//
//    advertise(O&o);           
//    advertise(O&o, const char *mode, int ms = 100);
//
//    payload(O&o,const char *mode)       // set advertising payload
//
// Arguments:
//
//    o:     Blob object (to avoid name clashes)
//    mode:  String containing a sequence of mode characters defining both
//           advertising type (upper case characters) and advertising flags 
//           (lower case characters), some of which have meaning, all others
//           are ignored and can be used for spacing (e.g. ':', ' ', '-', ...)
//    ms:    Advertising interval in mili seconds (default 100 ms)
//
// Advertising Type 
//
//    C:    ADV_CONNECTABLE_UNDIRECTED
//    D:    ADV_CONNECTABLE_DIRECTED
//    S:    ADV_SCANNABLE_UNDIRECTED
//    N:    ADV_NON_CONNECTABLE_UNDIRECTED
//
// Advertising Flags:
//
//    l:   LE Limited Discoverable Mode           (value: 0x01)
//    g:   LE General Discoverable Mode           (value: 0x02)
//    n:   BR/EDR Not Supported                   (value: 0x04)
//    c:   Controller: simultaneous LE & BR/EDR   (value: 0x08)
//    h:   Host: Simultaneous LE & BR/EDR         (value: 0x10)
//    
// Examples 1: Start advertising in peripheral role with type 'connectable, uni-
// directed, and flags 'BR/EDR Not Supported', 'LE General Discoverable Mode'
//
//    advertise(o,"C:ng",100);   // start advertising @ 100 msec interval
//    advertise(o,"Cng",100);    // same as above
//    advertise(o,"gCn",100);    // same as above
//
// Examples 2: Start advertising in peripheral role with type 'connectable,
// directed, and flags 'LE Limited Discoverable Mode'
//
//    advertise("D:l",100)       // Connectable Directed, limited discoverable
//
// Examples 3: A typical peripheral advertising session starts with setup of
// device name, advertising name and advertising data.
//
//    device(o,"Smart Button");        // setup device name
//    name(o,"Smart Button #1");       // setup advertising name
//    data(o,"My Home");               // setup advertising data
//    advertise(o,"C:ng",100);         // start advertising @ 100 msec interval
//
// Example 4: restart advertising based on pre-defined advertising payload
//
//
// See also: SERVICE
//
#ifndef _ADVERTISE_H_
#define _ADVERTISE_H_

#include "bricks/blob.h"
#include "bricks/trace.h"

#define _GADAT GapAdvertisingData
#define _GAPAR GapAdvertisingParams


   inline void payload(O&o,const char *p)   // Set advertising type and flags
   {
      _GAPAR::AdvertisingType_t typ = _GAPAR::ADV_CONNECTABLE_UNDIRECTED;
      uint8_t mask = 0x00;
      
      for (;*p; p++)
      {
          switch (*p)
          {
             case 'l':  mask = mask | _GADAT::LE_LIMITED_DISCOVERABLE;  break;
             case 'g':  mask = mask | _GADAT::LE_GENERAL_DISCOVERABLE;  break;
             case 'n':  mask = mask | _GADAT::BREDR_NOT_SUPPORTED;      break;
             case 'c':  mask = mask | _GADAT::SIMULTANEOUS_LE_BREDR_C;  break;
             case 'h':  mask = mask | _GADAT::SIMULTANEOUS_LE_BREDR_H;  break;
                
             case 'C':  typ = _GAPAR::ADV_CONNECTABLE_UNDIRECTED;       break;
             case 'D':  typ = _GAPAR::ADV_CONNECTABLE_DIRECTED;         break;
             case 'S':  typ = _GAPAR::ADV_SCANNABLE_UNDIRECTED;         break;
             case 'N':  typ = _GAPAR::ADV_NON_CONNECTABLE_UNDIRECTED;   break;
          }
      }
      
      o.gap().setAdvertisingType(typ);
      o.gap().accumulateAdvertisingPayload(mask);
   }


   inline void advertise(O&o)          // start advertising
   {
      o.gap().startAdvertising();      // start advertising
      trace(o,2,"<advertising>\n");
   }

   inline void advertise(O&o, int ms)  // start advertising (msec: periode)
   {
      o.gap().setAdvertisingInterval(ms);
      advertise(o);                    // start advertising
   }

   inline void advertise(Blob &o, const char *mode, int ms = 100)
   {
      payload(o,mode);                 // set advertising payload type & mask
      advertise(o,ms);                 // start advertising
   }

#endif // _ADVERTISE_H_