// blinker.cpp - send a morse pattern to LED1
//
// Function morse() is one way for running LED1 with a blinking sequence using
// a busy wait, until the sequence is completed.
//
//    Blinker b;
//    b.morse(" x xxx x    ");      send one time morse sequence, interval = 0.2
//    b.morse(" x xxx x    ",0.5);  send one time morse sequence, interval = 0.5
//
// Function morse(o) got the additional feature to stop an ongoing timer based
// blinking sequence.
//    
// The alternative is to setup an ever repeating blink sequence via LED1 using
// function blink(), which is non waiting.
//
//    Blinker b;
//    b.blink(" x xxx x    ");         repeating blink sequence, interval = 0.2
//    b.blink(" x xxx x    ",0.5);     repeating blink sequence, interval = 0.5
//    b.blink();                       stops blinking sequence
//

#include "bricks/target.h"
#include "bricks/blinker.h"

#ifndef LED_INVERTED
#   define LED_ON  1
#   define LED_OFF 0
#else
#   define LED_ON  0
#   define LED_OFF 1
#endif

#ifndef BLINK_LED
#   define BLINK_LED LED1
#endif

   static DigitalOut led(BLINK_LED);    // being used for morse sequence
   static Ticker ticker;                // triggers periodic callbacks
   static const char *pointer = 0;      // 0 means morse activity disabled
   static const char *sequence = 0;     // next morse sequence for repeats

   void Blinker::morse(const char *pattern, double interval)
   {
       pointer = 0;                     // disable ticker based blinking
       sequence = 0;                    // set also empty sequence
       
       for (; *pattern; pattern++)
       {
          led = (*pattern == ' ') ? LED_OFF : LED_ON;
          wait(interval);               // busy waiting for interval time
       }
   }
          
// callback for LED1 ticker controlled blinking   
           
   static void cbBlinker(void)         // blinker callback
   {
       if (pointer != 0)
      {
         if (*pointer == 0)
         {
            pointer = sequence;        // reset pointer to followup sequence
         }
        
         if (*pointer)
         {
            led = (*pointer++ == ' ') ? LED_OFF : LED_ON;
         }
      }
   }
   
   void Blinker::blink(const char *pattern, const char* next, double interval) 
   {
      pointer = 0;                      // stop current activities
      led = LED_OFF;                    // reset led with LED_OFF
      
      sequence = next;                  // init morse sequence   
         
      ticker.attach(cbBlinker,interval);// next LED state after every interval
      pointer = pattern;                // enable callback activty
   }  

   void Blinker::blink(const char *pattern, double interval)
   {
       blink(pattern,pattern,interval);
   }
   
   