#include "mbed.h"
#include "rtos.h"
#include "LCD_DISCO_F429ZI.h"


Serial bluetooth(PA_9, PA_10);    //Bluetooth Transmitter , Receiver Pins
AnalogIn ain(PA_0);               //Potentiometer Analog Input 
LCD_DISCO_F429ZI lcd;             //LCD Library

DigitalOut led1(LED1);            //LED Outputs
DigitalOut led2(LED2);


Thread thread1;                  //Instantiating threads
Thread thread2;
Thread thread3;
Thread thread4;

typedef struct {
    float    voltage;      
    float    current;      
} results;                         //struct type for the results

MemoryPool<results, 1> mpool;     //Creating a memory pool for queue
Queue<results, 1> queue;          //Queue for data transfer

Semaphore one_slot(1);             //Instantiating a single semaphore resource slot


 


/*The following functions gets the ADC data & sends the results to the receiver end via memory queue
Input Parameter : Thread Name */

void data_send(void const *name) {    
    lcd.Clear(LCD_COLOR_WHITE);                                                      //LCD Background Clear
    lcd.DisplayStringAt(0, LINE(1), (uint8_t *)"Sending Data ... ", CENTER_MODE);   //Display info on LCD    
    
while(1) {
    led1 = !led1;            //LED1 toggle
    bluetooth.printf("\n(%s has received the data from ADC, now sending it via Queue...)\n\r", (const char *)name);
    
    //Thread-1 reads the ADC Data
    results *message = mpool.alloc();           //allocating fixed size memory        
    message->voltage = ain.read()*3;           //reference voltage = 3v
    message->current = (ain.read()*3)/100;     // voltage/resistance (100 ohm)
    queue.put(message);                        //Adding data to the queue
    Thread::wait(2000);   
}
}


/*The following function receives the data from the memory queue &  
transmits the data to be displayed over the phone using the Bluetooth module
Input Parameter : Thread Name */

void data_receive(void const *name) {
    lcd.DisplayStringAt(0, LINE(2), (uint8_t *)"Data Received !!", CENTER_MODE); //Display info on LCD 
      
while(1) {
    led2 = !led2;         //LED2 toggle
    
    one_slot.wait();      //wait until a semaphore resource becomes available
    
    bluetooth.printf("\n---------------------------------------------------------------------\n");
    bluetooth.printf("%s has acquired the semaphore & received the data.\n\r", (const char *)name);
    bluetooth.printf("\nResults are now displayed by %s\n\r", (const char *)name);
    Thread::wait(1000);

    /*Receive queue data and display via Bluetooth*/

    osEvent evt = queue.get(); // get a message from the queue
    
    //check for the message
    if (evt.status == osEventMessage) {
     results *message = (results*)evt.value.p;
     bluetooth.printf("\nVoltage: %.2f V\n\r"   , message->voltage);
     bluetooth.printf("Current: %.2f A\n\r"     , message->current);
     mpool.free(message);}

    one_slot.release();       //release the semaphore resource
    bluetooth.printf("\n%s releases semaphore\r", (const char *)name);
    Thread::wait(1000);
          
}
}


int main() {
    printf("We are in main.\n\r");
    
    thread1.set_priority(osPriorityHigh);               //setting Task Priorities
    thread2.set_priority(osPriorityNormal);
    thread3.set_priority(osPriorityNormal);

    thread1.start(data_send, (void *)"Thread 1");       // Starting the processes
    thread2.start(data_receive, (void *)"Thread 2");
    thread3.start(data_receive, (void *)"Thread 3");
     
    thread1.join();                                    //synchronization of threads
    thread2.join();
    thread3.join();
    
    //wait for threads to complete
    printf("The end of main.\n\r");
}
    
 
