// A high prio thread reads at a rate of 1kHz from a SRF08 ultrasonic ranger
// data packets of different size, whereas the lower prio main thread measures the CPU time left.

#include "mbed.h"
#include "rtos.h"
#include "I2CMasterRtos.h"
#include "stdint.h"

volatile int g_disco=0;
volatile int g_len=2;
volatile int g_freq=100000;

I2CMasterRtos i2c(p28, p27);
const uint32_t adr = 0x70<<1;

void highPrioCallBck(void const *args)
{
    i2c.frequency(g_freq);
    // read back srf08 echo times (1+16 words)
    const char reg= 0x02;
    char result[64];
    uint32_t t1=us_ticker_read();
    i2c.read(adr, reg, result, g_len);
    uint32_t dt=us_ticker_read()-t1;
    uint16_t dst=((static_cast<uint16_t>(result[0])<<8)|static_cast<uint16_t>(result[1]));
    if(--g_disco>0) printf("dst=%4dcm dt=%4dus\n",dst,dt);
}

int doit()
{
    RtosTimer highPrioTicker(highPrioCallBck, osTimerPeriodic, (void *)0);

    Thread::wait(500);
    // trigger srf08 ranging
    const char regNcmd[2]= {0x00,0x54};
    i2c.write(adr, regNcmd, 2);
    osDelay(200);

    highPrioTicker.start(1);

#if defined(TARGET_LPC1768)
    const int nTest=7;
    const int freq[nTest]=  {1e5,   1e5,    1e5,   4e5,    4e5,    4e5,    4e5};
    const int len[nTest]=   {1,     4,      6,      1,     6,     12,      25};
#elif defined(TARGET_LPC11U24)
    const int nTest=5;
    const int freq[nTest]=  {1e5,   1e5,    4e5,   4e5,    4e5    };
    const int len[nTest]=   {1,     6,      1,      6,     23};
#endif
    for(int i=0; i<nTest; ++i) {
        g_freq = freq[i];
        g_len = len[i];
        printf("f=%d l=%d\n",g_freq,g_len);
        Thread::wait(500);
        const uint32_t dt=1e6;
        uint32_t tStart = us_ticker_read();
        uint32_t tLast = tStart;
        uint32_t tAct = tStart;
        uint32_t tMe=0;
        do {
            tAct=us_ticker_read();
            #if defined(TARGET_LPC1768)
            if(tAct==tLast+1)++tMe;
            #elif defined(TARGET_LPC11U24)
            uint32_t delta = tAct-tLast;
            if(delta<=2)tMe+=delta;  // on the 11U24 this loop takes a bit longer than 1µs (ISR ~3µs, task switch ~8µs)
            #endif
            tLast=tAct;
        } while(tAct-tStart<dt);
        printf("dc=%5.2f \n", 100.0*(float)tMe/dt);
        g_disco=5;
        while(g_disco>0);
    }
    return 0;
}