#ifndef FTENCODER_H
#define FTENCODER_H

#include "InterruptIn.h"
#include "us_ticker_api.h"
#include "FunctionPointer.h"
#include "TimeoutTweaked.h"

using namespace mbed;

/// Simple Interface for the simple encoder of the red fischertechnik motors.
/// The interface counts both, rising and falling edges, resulting in 150 counts per revolution.
/// The interface also provides a speed measurement updated with each edge. 
/// @note Connect the green wire to GND, the red one to +3.3V and the 
/// black signal line to any of mbed numbered pins. Additionally connect the signal line
/// via a pullup resitor to +3.3V. A 10K resistor works fine.
/// 
class FtEncoder
{
    static const float c_speedFactor = 1e6/75.0; /// 1/(µs * 150/2) edges per revolution)
    static const int c_nTmStmps = 1<<2; /// size of ringbuffer: Has to be 2^x

    InterruptIn m_encIRQ;
    TimeoutTweaked m_tmOut;

    volatile unsigned int m_cnt; /// edge counter
    unsigned int m_standStillTimeout; 
    volatile unsigned int m_cursor;  /// ringbuffer cursor
    volatile uint32_t m_timeStamps[c_nTmStmps]; /// ringbuffer of edge timestamps 

    FunctionPointer m_callBack;

    /// ISR called on rising and falling encoder edges
    void encoderISR();
    /// ISR called on edge timeout i.e. standstill
    void timeoutISR();

public:

    /// create a simple interface to the encoder connected to the given pin
    /// @param name of the pin the black encoder signal wire is connected to
    /// @param standStillTimeout After this time [µs] without any encoder edge getSpeed() returns zero
    FtEncoder(PinName pwm, unsigned int standStillTimeout=50000);

    /// get number of falling and rising encoder edges
    inline unsigned int getCounter() const {
        return m_cnt;
    };

    /// reset the encoder edge counter to the given value (default is 0)
    inline void resetCounter(unsigned int cnt=0) {
        m_cnt=cnt;
    };

    /// get period [µs] between the two latest edges of same direction (rising or falling)
    /// @note Returns standStillTimeout if motor stands still or no edges have been detected yet
    unsigned int getLastPeriod() const;

    /// get time stamp [µs] of last update i.e. encoder edge or timeout
    unsigned int getTimeStampOfLastUpdate() const {
        return m_timeStamps[m_cursor];
    };

    /// returns current time stamp [µs]
    /// @note simply calls us_ticker_read() from mbed's C-API (capi/us_ticker_api.h)
    inline unsigned int  getCurrentTimeStamp() const {
        return us_ticker_read();
    }

    /// get speed in revolutions per second (Hz)
    float getSpeed() const;

    /// hook in a call back to the encoder ISR e.g. for sending a RTOS signal
    void setCallBack(FunctionPointer callBack) {
        m_callBack=callBack;
    };
};

#endif