#ifndef FTCONTROLSETSIGNAL_H
#define FTCONTROLSETSIGNAL_H

#include <cstdint>
#include "mbed.h"

/// interface class that translates the raw message of ft IR remote control
/// @note just a wrapper arround an evil bitfield, not really portable, ... but they are so handy
class FtControlSetMessage
{
    union { 
        //  message structure: 30 payload bits plus 2 status bit added by receiver ISR plus 
        struct { 
            unsigned status:2;      // @see Status
            unsigned tail:1;        // should be 0
            unsigned parity:1;      // set if parity even
            unsigned leftXDir:1;    // direction left x
            unsigned leftYDir:1;    // direction left y
            unsigned rightXDir:1;   // direction right x
            unsigned rightYDir:1;   // direction right y
            unsigned leftXSpd:4;    // speed left x
            unsigned leftYSpd:4;    // speed left y
            unsigned rightXSpd:4;   // speed right x
            unsigned rightYSpd:4;   // speed right y
            unsigned onBut:1;       // ON button
            unsigned offBut:1;      // OFF button
            unsigned switch1:1;     // switch 1
            unsigned switch2:1;     // switch 1
            unsigned header:4;      // should be 8;
        };
        uint32_t m_rawMessage;
    };

public:

    /// Status enum returned by getStatus().
    enum Status {/// everything is fine
        OK=0,       /// wrong header ID
        FrameKO,  /// parity error
        ParityKO  
        //ReceiveKO=1,/// message was corrupted: Wrong number of pulses or bad timing
    } ;

    /// Constructor from raw 32bit message
    FtControlSetMessage(uint32_t rawMessage):m_rawMessage(rawMessage) {};

    /// get x position of left control stick [-15,15]
    int32_t getLeftX()const {
        return leftXDir?leftXSpd:-leftXSpd;
    };
    /// get y position of left control stick [-15,15]
    int32_t getLeftY()const {
        return leftYDir?leftYSpd:-leftYSpd;
    };
    /// get x position of right control stick [-15,15]
    int32_t getRightX()const {
        return rightXDir?rightXSpd:-rightXSpd;
    };
    /// get y position of right control stick [-15,15]
    int32_t getRightY()const {
        return rightYDir?rightYSpd:-rightYSpd;
    };
    /// get state of the ON button
    bool getOnButton()const {
        return onBut;
    };
    /// get state of the OFF button
    bool getOffButton()const {
        return offBut;
    };
    /// get state of switch 1 (Frequency setting)
    bool getSwitch1()const {
        return switch1;
    };
    /// get state of switch 2 (Receiver ID)
    bool getSwitch2()const {
        return switch2;
    };
    /// get Status of this message, see Status
    Status getStatus()const {
        return static_cast<Status>(status);
    };

    /// get message in raw format
    uint32_t getRawMessage()const {
        return m_rawMessage;
    };
};

#endif