/* Copyright (c) 2015 Liyang HU, MIT License
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
 * CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#include <algorithm>
#include <math.h>
#include <mbed.h>
#include <Pulsator.h>

/*! \class Pulsator
 * \brief Pulsate an LED using hardware \a PwmOut and \a Ticker.
 * \code
#include <mbed.h>
#include <Pulsator.h>

Pulsator led(LED1);

int main(void) {
    led.period(2.0).gamma(1.8) = true;
    while(1) wait(1);
}
 * \endcode
 * \see \a PwmOut, \a Ticker
 */

void Pulsator::disable(void)
{
    _enable = false;
    detach();
    out = _active_high ? 0.0f : 1.0f;
}

void Pulsator::enable(void)
{
    out.period(1.0f / 1024.0f);
    phase = 0.0f;
    _enable = true;
    step();
    attach(this, &Pulsator::step, 0.5f * _period / (float)(_levels - 1));
}

//! Bit of a hack to update _delay without re-attaching the handler.
void Pulsator::reload(void)
{
    _delay = 1000000.0f * 0.5f * _period / (float)(_levels - 1);
}

void Pulsator::step(void)
{
    float x = _fun(phase);
    if(x < 0.0f) x = 0.0f;
    if(x > 1.0f) x = 1.0f;
    float level = powf(x, _gamma);
    out = _active_high ? level : 1.0f - level;
    phase += 0.5f / (float)(_levels - 1);
    if(phase >= 1.0f)
        phase -= 1.0f;
}

//! Create a \a Pulsator attached to the specified \a pin.
//! \param pin Pin to pulsate.
//! \note Some platforms can only drive certain pins with the hardware PWM.
Pulsator::Pulsator(PinName pin) : out(pin)
{
    active_high(); disable(); fun(); gamma(); period(); levels();
}

//! Enable or disable the output.
/* \note Passing \a false deactivates the output completely, rather than
    maintaining the LED brightness at the point when this is called.
    Conversely, \a true starts pulsating from the inactive state.
    Setting the curent state is a no-op. */
Pulsator& Pulsator::operator=(bool state)
{
    if(state != _enable)
        state ? enable() : disable();
    return *this;
}

//! Get the current state of the output.
Pulsator::operator bool(void)
{
    return _enable;
}

//! Should the output pin be active high (\a true), or active low (\a false)?
Pulsator& Pulsator::active_high(bool high)
{
    _active_high = high;
    if(!_enable)
        out = _active_high ? 0.0f : 1.0f;
    return *this;
}

#define TAU 6.2831853f
static float sinusoidal(float phase) { return 0.5f - 0.5f * cosf(TAU * phase); }

//! Set the intensity function.
/* \param fp Pointer to a function returning an intensity
    in the range [0,1]. Values outside will be clamped. */
//! \param phase Input to \a fp, in the range [0, 1).
//! \note Pass \a NULL for the default sinusoidal function.
Pulsator& Pulsator::fun(float (*fp)(float phase))
{
    _fun = fp ? fp : &sinusoidal;
    return *this;
}

//! Set the gamma correction for the output LED.
Pulsator& Pulsator::gamma(float power)
{
    _gamma = power;
    return *this;
}

//! Set the number (>= 2) of distinct brightness levels.
Pulsator& Pulsator::levels(uint16_t number)
{
    _levels = number;
    reload();
    return *this;
}

//! Set the duration of each cycle, in seconds.
Pulsator& Pulsator::period(float seconds)
{
    _period = seconds;
    reload();
    return *this;
}
