#ifndef STATE_H
#define STATE_H

#include "N5110.h"
#include "PinDetect.h"
#include "InputManager.h"
#include "Music.h"
#include "ShiftReg.h"

/** @file State.h
* @author Andreas Garmannslund
* @date April 2015
*/

class StateManager;

/// States used in the finite state machine.
enum MainState {MAIN_MENU, GAME, SUBMIT_HIGHSCORE, GAME_OVER, NO_STATE, TITLE_SCREEN};

/// Abstract class for states in the program's main finite state machine. All state implementations is derived from this abstract class.
class State
{   
    public:
        
        /* Creates a new state object. Should be called from child's constructor.
        * @param fsm Pointer to finished state machine.
        * @param lcd Pointer to the N5110 lcd object.
        * @param input Pointer to the InputManager object, used for controlling user input.
        */
        State(StateManager* fsm, N5110 *lcd, InputManager* input, Sound* sound, ShiftReg* shiftreg)
                :lcd(lcd), input(input), sound(sound), shiftreg(shiftreg), fsm(fsm) {}
                
        /// Handle user input and update logic.
        virtual void update(float dt) = 0;
        
        /// Draw to screen.
        virtual void render() = 0;
        
    protected:
        /* Requests the finite state machine to switch to a new state when possible.
        * @param newState The state the fsm should switch to.
        **/
        void requestStateChange(MainState newState);
        
        /** Draws an image to the lcd
        * @param img Array with the same size as the display, where 1 is opaque, 0 is blank.
        */
        //void drawImage(const int img[BANKS][WIDTH]); // Draws an image from array
        
        
        /** Draws an image/sprite to the lcd
        *   Only the solid pixels are drawn. If two images overlap, the second image drawn will
        *   not clear pixels which are solid in the first image.
        *   @param img const int array where a solid pixel equals 1, and a blank pixel equals zero
        *   @param x Horizontal position of image (leftmost pixel)
        *   @param y Vertical position of image (uppermost pixel)
        *   @param Inverses images. Default value is false
        *   See seperate program for how this array can be generated from an image file using SFML!
        */
        template<class T, size_t rows, size_t cols>
        void drawImage(const T (&img)[rows][cols], int x = 0, int y = 0, bool inverse = false, bool flipX = false, bool flipY = false);
        
    protected:
        N5110 *lcd;
        InputManager *input;
        Sound *sound;
        ShiftReg *shiftreg;
        
    private:
        StateManager *fsm;
        
};

// Template functions needs to be declared in the header file
// TODO: Add functions for inverse drawing
template<class T, size_t rows, size_t cols>
void State::drawImage(const T (&img)[rows][cols], int x, int y, bool inverse, bool flipX, bool flipY)
{
    int targetX, targetY; // The position on the lcd we are writing to
    
    for (int i = 0; i < rows; ++i)
    {
        targetY = (flipY) ? (y + (rows-1) - i) : (y + i);
        
        // Skip if outside dimensions of LCD
        if (targetY < 0) continue;
        else if (targetY >= HEIGHT) continue;
        
        for (int j = 0; j < cols; ++j)
        {
            targetX = (flipX) ? (x + ((cols - 1) - j)) : (x + j);
            
            // Dimensions check. Draws left to right.
            if (targetX < 0) continue;
            else if (targetX >= WIDTH) continue;
            
            int solid = img[i][j]; 
            
            if ((solid && !inverse) || (!solid && inverse))
                lcd->setPixel(targetX, targetY);
                
        }
    }   
}
#endif