/*
* @file main.cpp
* @brief BRAVEHEART -- 2D dungeon game in XJEL2645 Project
* @author Yadong Hu
* @date March, 2021
*/

/// libs
#include "mbed.h"
#include "N5110.h"
#include "PinDetect.h"
#include "ShiftReg.h"
#include <string>
#include <sstream>
#include <ctime>

/// customize libs
#include "StateManager.h"
#include "State.h"
#include "InputManager.h"
#include "Music.h"

/// pin specification
#define JOY_H p19
#define JOY_V p20
#define JOY_BTN p17
#define LIGHT_SENSOR p15
#define BUTTON_A p29
#define BUTTON_B p28
#define BUTTON_C p27
#define BUTTON_D p26

/// global variables
int seven_seg_array [] = {
    0x3F,0x06,0x5B,0x4F,0x66,0x6D,0x7D,0x07,0x7F,0x6F,0x77,0x7C,0x39,0x5E,0x79,0x71
};

/// I/O specification
// screen
N5110 *lcd;

// create ShiftReg object
ShiftReg *shiftreg;  
ShiftReg shift;

// background music
Sound *sound; 

// inputs assemble
InputManager *input;

// light brightness detector
AnalogIn light_sensor(LIGHT_SENSOR);

// to generate rand seeds from noise
AnalogIn randomPin(p16); 
AnalogIn randomPin2(p17);

/// Finite state machine
StateManager* fsm;

/// Access file system
LocalFileSystem local("local");

/// Initialize
void init();

/// Generate seeds
unsigned int getRandSeed();

/// Access the records file
void readHighscoreFile();

int main()
{    
    srand(getRandSeed());
    init();
    Timer timer;
    timer.start();
    while(true)
    {
        while (timer.read() < 0.08)
            sleep();

        // update
        lcd->setBrightness(1.0 - light_sensor); // Update brightness of screen
        fsm->update(timer.read());
        input->joystick->update();
        
        // render
        lcd->clear();
        fsm->render(); 
        lcd->refresh();
        
        fsm->processRequest(); // Change the state if requested.
        
        timer.reset();
    }
    
}

void init()
{
    // Init LCD
    lcd = new N5110(p14, p8, p9, p10, p11, p13, p21);
    lcd->init();
    lcd->normalMode();
    lcd->setBrightness(1.0 - light_sensor); // Update brightness of screen
    
    // write 0 to 7-seg to turn it off
    shift.write(0x00);
    
    // Input
    input = new InputManager(BUTTON_A, BUTTON_B, BUTTON_C, BUTTON_D, JOY_H, JOY_V, JOY_BTN);

    // Finite state machine
    fsm = new StateManager(lcd, input, sound, shiftreg, TITLE_SCREEN);
    
    readHighscoreFile(); // load highscores from file system
    sound = new Sound(p18);
    sound->playNote(SFX::RESTART);
}

void readHighscoreFile()
{    
     FILE *fp = fopen("/local/highscores.txt", "r");    // read from local file
     
     if (!fp)       // if file is not created yet the highscores will default values as defined in Global.cpp
        return;
        
     char initials[4];
     char scoreStr[8];

     for (int i = 0; i < 3; ++i)
     {   
         fscanf(fp, "%s %s", initials, scoreStr);   // read from file
         
         int score = atoi(scoreStr);    // convert to int
         
         // update global highscore table
         Global::highscores[i].initials = initials;
         Global::highscores[i].score = score;
     }
        
     fclose(fp);
}


/** Get a random seed based on unconnected analog input pins
  * @see https://developer.mbed.org/questions/2886/Why-is-the-rand-function-not-the-least-b/
 */
unsigned int getRandSeed()
{
    unsigned int randNum = 0;
    unsigned int lsb; // least significant bit
    
    for (unsigned int i = 0; i < 16; ++i)
    {
        lsb = 1 & (randomPin.read_u16()); // Read least significant bit in randomInput - value depends on noise
        randNum |= (lsb << i);              // Set bit nr.i to lsb in randomInput
        
        lsb = 1 & (randomPin2.read_u16());
        randNum |= (lsb << (16+i));
    }
    
    return randNum;
}