#ifndef GLOBAL_H
#define GLOBAL_H

#include "mbed.h"
#include <string>
#include <vector>

/// @file Global.h
/// @brief Contains global variables and functions

namespace Record
{
    /// A highscore element consists of initials and a score.
    struct Highscore
    {
        std::string initials;
        int score;
    };
    
    /// An achievement element consists of all records made.
    struct Achievement
    {
        int totalGamePlayed;
        int totalSucceedTimes;
        int totalDeathTimes;
        int totalWalkLength;
        int totalChestUnlocked;
        int totalCoinSpent;
        int totalEnemyDiscovered;
        int totalWeaponDiscovered;
        int totalCharacterDiscovred;
        int totalEnemyDestroyed;
        int totalBulletShot;
        int totalBulletAccuracy;
    };
}

/// Simple objects with a position (x,y) and velocity (vx, vy)
struct Point
{
    int x;
    int y;
    int vx, vy;
};

/// Rectangle with position (x and y) and dimensions (width and height)
struct Rectangle
{
    Rectangle(int x, int y, int w, int h) : x(x), y(y), width(w), height(h) {}
    int x;
    int y;
    int width;
    int height;
};

namespace Global
{
    /// Score in last game
    extern int score;
    
    /// List showing the three all time highest score achieved. Sorted from highest to lowest.
    extern Highscore highscores[3];     
    
    /// Clears the highscore list
    void clearHighscoreList();
}

/** An entity represents a movable character, such as the player, enemies etc.
*   Note that the entity class does not contain the sprite (image) of the entity.
*   Different sprites are given as 2D const int arrays in 
*   OBS! The entity's dimensions should be the same as the width and height or else this will lead to undefined behaviour!
*/
class Entity
{
    public:
        Entity() {x = y = width = height = vx = vy = 0; facingLeft = true; dead = false;}
        Entity(int x, int y, int w = 0, int h = 0) : x(x), y(y), width(w), height(h) {vx = vy = 0; facingLeft = true; dead = false;}
        
        /// Position of entity (origin: left upper corner)
        int x, y;
        
        /// Velocity of entity
        int vx, vy;
        
        /// Width of entity
        int width; 
        
        /// Height of entity     
        int height;
        
        /// True if the entity is facing left
        bool facingLeft; 
        
        /// True if enemy is dead.
        bool dead;
        
        /// Returns x-position of the right edge
        int getRight() {return x + width - 1;}
        
        /// Returns y-position of the bottom edge   
        int getBottom() {return y + height - 1;}
};

/// Enemy class
class Enemy : public Entity
{
    public:
        /// Different type of enemies
        enum Type{SIMPLE, JUMPER, RUNNER};
    
    public:
        Enemy(int x, int y, bool facingLeft, Type type = SIMPLE) : Entity(x,y), type(type) {setup();}
        
        /// What type of enemy it is.
        Type type;      
        
        /// Multiplier used for giving more points
        int difficulty; 
        
        /// Probability (in percent) of trying to jump if at ground.
        int jumpRate;
        
        private:
            void setup();
};

namespace Map
{
    enum MapType {Normal, Store, Chest, BIG, BOSS};
    struct MapProperty
    {
        int mapSize[2];
        int respawnPosition[2];
        int succeedPosition[2];
        int enermyPosition[3][2];
        MapType mapType;
        int chestPosition[2];
        int storePosition[6];
        int goldCoinNum;
        int goldCoinPosition[10];
        int spawnRate;
        int maxEnermy;
    };
}

namespace GameX
{
    struct GameProperty
    {
        Entity player;  // Player object
        std::vector<Point*> bullets;    // Container for bullets
        std::vector<Enemy*> enemies;    // Container for enemies
        int livesLeft;  // The number of lives left
        bool paused;    // True if the game is paused
        int currentLayer; // To determine which layer the player is.
    };
}

#endif