#ifndef INPUT_H
#define INPUT_H

#include "N5110.h"
#include "PinDetect.h"
#include "InputManager.h"

/// @file InputManager.h


/** @brief Enum used for the 8-directions of the joystick. */
enum JoystickDirection {CENTER, UP, DOWN, LEFT, RIGHT, UP_LEFT, UP_RIGHT, DOWN_LEFT, DOWN_RIGHT, UNKNOWN};

/// Joystick class
class Joystick
{   
    public:
    
        /** Creates a new Joystick object
        * @param x Pin connected to the horizontal potentiometer of the joystick
        * @param y Pin connected to the vertical potentiometer of the joystick
        * @param button Pin connected to the button of the thumb joystick
        */
        Joystick(PinName x, PinName y, PinName button);
        
        /** @brief Deconstructor. Frees allocated memory */
        ~Joystick();
        
        /** @brief Updates the current direction and button status of the joystick */
        void update();
        
        /** Calibrates the joystick. The joystick must be centered while this function is called */
        void calibrate(); // Calibrates joystick by updating the center positions
        
        /** Returns the current JoystickDirection based on last update
        * @return The current JoystickDirection.
        */
        int getDirection() {return dir;}
        
        /** Reads the value of the button
        * @return 1 if pressed, 0 otherwise
        */
        int readButton() {return *btn;};
        
        /** Square set around the center of the joystick where the input is ignored. The axes are treated seperately. Can be varied from 0 (no dead-zone) to 0.5 (max value for dx and dy). **/
        static const float DEAD_ZONE = 0.1; 
        
    private:
        AnalogIn *xPot;
        AnalogIn *yPot;
        DigitalIn *btn;
        float dx, dy;
        float centerX, centerY;
        JoystickDirection dir;
};

struct Input
{
    /** Used as identificator for the different buttons */
    enum Button{ButtonA, ButtonB, ButtonC, ButtonD};
};

/// Used to manage user input from buttons and thumb joystick
class InputManager
{   
    public:
    
        /** Creates a new InputManager object
        * @param pinA Pin connected to button A
        * @param pinB Pin connected to button B
        * @param pinc Pin connected to button C
        * @param pind Pin connected to button D
        * @param x Pin connected to the horizontal potentiometer of the joystick
        * @param y Pin connected to the vertical potentiometer of the joystick
        * @param button Pin connected to the button of the thumb joystick
        */
        InputManager(PinName pinA, PinName pinB, PinName pinC, PinName pinD, PinName joyH, PinName joyV, PinName joyBtn);
        
        /** Deconstructor. Frees allocated memory related to the buttons and the joystick **/
        ~InputManager();
        
        Joystick *joystick;
        
        /** @brief Adds a button interrupt which is invoked when the button is pressed. Button needs to be released for the interrupt to occur again.
        * @param button Name of the button.
        * @param func Callback function.
        */
        void addBtnPressInterrupt(Input::Button button, void (*func)(void));
        
        
        /** Reads the current value of a button.
        * @param button The button we want to read.
        * @return Returns 1 if button is pressed, 0 otherwise.
        */
        int read(Input::Button button);
    
    private:
        /// Button objects
        PinDetect *btnA;
        PinDetect *btnB;
        PinDetect *btnC;
        PinDetect *btnD;
        
        /** Returns a pointer to the actual button object
        * @param button The requested button.
        * @return Pointer to the button.
        */
        PinDetect* getBtnPtr(Input::Button button);      
};

#endif

