#include "InputManager.h"

/** @file InputManager.cpp
 *  @author Andreas Garmannslund
 *  @date April 2015
*/

InputManager::InputManager(PinName pinA, PinName pinB, PinName pinC, PinName pinD, PinName joyH, PinName joyV, PinName joyBtn)
{
    // Init joystick
    joystick = new Joystick(joyH, joyV, joyBtn);
    joystick->calibrate();
    
    // Init buttons
    btnA = new PinDetect(pinA);
    btnB = new PinDetect(pinB);
    btnC = new PinDetect(pinC);
    btnD = new PinDetect(pinD);
    
    // Default sample frequency
    btnA->setSampleFrequency();
    btnB->setSampleFrequency();
    btnC->setSampleFrequency();
    btnD->setSampleFrequency();
}

InputManager::~InputManager()
{
    delete joystick;
    delete btnA;
    delete btnB;
    delete btnC;
    delete btnD;
}

void InputManager::addBtnPressInterrupt(Input::Button button, void (*func)(void))
{
    PinDetect *btn = getBtnPtr(button);
    if (btn) // if not null pointer
        btn->attach_asserted(func);
}

PinDetect* InputManager::getBtnPtr(Input::Button button)
{
    switch(button)
    {
        case Input::ButtonA:
            return btnA;
        
        case Input::ButtonB:
            return btnB;
        
        case Input::ButtonC:
            return btnC;
            
        case Input::ButtonD:
            return btnD;
        
        default:
            return 0; // Return 0 (nullptr) if invalid input
    }
}

int InputManager::read(Input::Button button)
{
    PinDetect *btn = getBtnPtr(button);
    if (!btn) return 0; // Invalid button
    
    return *btn; // Value of button
}

Joystick::Joystick(PinName x, PinName y, PinName button)
{
    xPot = new AnalogIn(x);
    yPot = new AnalogIn(y);
    btn = new DigitalIn(button);
}

Joystick::~Joystick()
{
    delete xPot;
    delete yPot;
    delete btn;
}


// Make sure that the Joystick is centered while calibrating
void Joystick::calibrate()
{
    btn->mode(PullDown);
    centerX = *xPot;
    centerY = *yPot;
}


void Joystick::update()
{
    // Get position of joystick compared to the center values (delta x and delta y).
    dx = *xPot - centerX;
    dy = *yPot - centerY;
    
    if (abs(dx) < DEAD_ZONE && abs(dy) < DEAD_ZONE)
        dir = CENTER;
    else if (abs(dx) < DEAD_ZONE)                       // Inside horizontal dead zone.
        dir = (dy > DEAD_ZONE) ? UP : DOWN;
    else if (abs(dy) < DEAD_ZONE)                       // Inside vertical dead zone.
        dir = (dx > DEAD_ZONE) ? LEFT : RIGHT;
    else if (dx > 0)                                    // To the left, outside both deadzones.
        dir = (dy > 0) ? UP_LEFT : DOWN_LEFT; 
    else                                                // To the right, outside both deadzones
        dir = (dy > 0) ? UP_RIGHT : DOWN_RIGHT;
}
