#ifndef _PWM_OUT_H_
#define _PWM_OUT_H_

#include "BCM2835.h"

class PwmOut {

public:

    /** Create a PwmOut connected to the specified pin
     *
     *  @param pin PwmOut pin to connect to
     */
    PwmOut(PinName pin = gpio18);

    ~PwmOut();

    /** Set the output duty-cycle, specified as a percentage (float)
     *
     *  @param value A floating-point value representing the output duty-cycle,
     *    specified as a percentage. The value should lie between
     *    0.0f (representing on 0%) and 1.0f (representing on 100%).
     *    Values outside this range will be saturated to 0.0f or 1.0f.
     */
    void write(float value);

    /** Return the current output duty-cycle setting, measured as a percentage (float)
     *
     *  @returns
     *    A floating-point value representing the current duty-cycle being output on the pin,
     *    measured as a percentage. The returned value will lie between
     *    0.0f (representing on 0%) and 1.0f (representing on 100%).
     *
     *  @note
     *  This value may not match exactly the value set by a previous write().
     */
    float read();

    /** Set the PWM period, specified in bcm2835PWMPulseWidth (micro/nano seconds), keeping the duty cycle the same.
     *  @note  Sets clock divider according to the required period.
     *  @param period Change the period of a PWM signal. The allowed values are:
     *         BCM2835_PWM_PERIOD_212_US  ->  213.33 us  =    4.6875 kHz
     *         BCM2835_PWM_PERIOD_107_US  ->  106.66 us  =    9.375  kHz
     *         BCM2835_PWM_PERIOD_53_US   ->   53.33 us  =   18.75   kHz
     *         BCM2835_PWM_PERIOD_27_US   ->   26.66 us  =   37.50   kHz
     *         BCM2835_PWM_PERIOD_13_US   ->   13.33 us  =   75.00   kHz
     *         BCM2835_PWM_PERIOD_7_US    ->    6.66 us  =  150.00   kHz
     *         BCM2835_PWM_PERIOD_3_US    ->    3.33 us  =  300.00   kHz
     *         BCM2835_PWM_PERIOD_2_US    ->    1.66 us  =  600.00   kHz
     *         BCM2835_PWM_PERIOD_833_NS  ->  833.33 ns  = 1200.00   kHz
     *         BCM2835_PWM_PERIOD_417_NS  ->  416.66 ns  = 2400.00   kHz
     *         BCM2835_PWM_PERIOD_208_NS  ->  208.33 ns  = 4800.00   kHz
     *         BCM2835_PWM_PERIOD_104_NS  ->  104.16 ns  = 9600.00   kHz
     */
    void period_ms(int period);
    void period_us(int period);
    void period_ns(int period);

    /** A operator shorthand for write()
     *  \sa PwmOut::write()
     */
    PwmOut &operator= (float value)
    {
        write(value);
        return *this;
    }

    /** A operator shorthand for write()
     * \sa PwmOut::write()
     */
    PwmOut &operator= (PwmOut &rhs)
    {
        write(rhs.read());
        return *this;
    }

    /** An operator shorthand for read()
     * \sa PwmOut::read()
     */
    operator float()
    {
        return read();
    }

protected:
    PinName     _pwmPin;
    float       _duty_cycle;
    uint32_t    _range;
    uint32_t    _period;
};

#endif	// _PWM_OUT_H_

