#include "mbed.h"

extern volatile uint32_t *bcm2835_bsc1;
extern volatile uint32_t *bcm2835_spi0;

/**************************************************************************
 *
 * SPI Class implementation
 *
 **************************************************************************/

/******************
 * Public methods *
 ******************/
SPI::SPI() :
    _write_fill(0xFF)
{
    REV = getBoardRev();

    bcm2835_gpio_fsel(9, BCM2835_GPIO_FSEL_ALT0);   // MISO
    bcm2835_gpio_fsel(10, BCM2835_GPIO_FSEL_ALT0);  // MOSI
    bcm2835_gpio_fsel(11, BCM2835_GPIO_FSEL_ALT0);  // CLK

    // Set the SPI CS register to some sensible defaults
    volatile uint32_t*  paddr = bcm2835_spi0 + BCM2835_SPI0_CS / 4;
    bcm2835_peri_write(paddr, 0);                   // All 0s

    // Clear TX and RX fifos
    bcm2835_peri_write_nb(paddr, BCM2835_SPI0_CS_CLEAR);
}

/**
 * @brief
 * @note
 * @param
 * @retval
 */
SPI::~SPI()
{
    // Set all the SPI0 pins back to input

    //    bcm2835_gpio_fsel(7, BCM2835_GPIO_FSEL_INPT);   // CE1
    //    bcm2835_gpio_fsel(8, BCM2835_GPIO_FSEL_INPT);   // CE0
    bcm2835_gpio_fsel(9, BCM2835_GPIO_FSEL_INPT);   // MISO
    bcm2835_gpio_fsel(10, BCM2835_GPIO_FSEL_INPT);  // MOSI
    bcm2835_gpio_fsel(11, BCM2835_GPIO_FSEL_INPT);  // CLK
}

/**
 * @brief
 * @note
 * @param
 * @retval
 */
void SPI::format(int bits, uint8_t mode)
{
    volatile uint32_t*  paddr = bcm2835_spi0 + BCM2835_SPI0_CS / 4;

    if (bits > 0) {
        bits = 8U;
    }
    //BCM2835_SPI_BIT_ORDER_MSBFIRST is the only one suported by SPI0

    // Mask in the CPO and CPHA bits of CS
    bcm2835_peri_set_bits(paddr, mode << 2, BCM2835_SPI0_CS_CPOL | BCM2835_SPI0_CS_CPHA);
}

/**
 * @brief
 * @note
 * @param
 * @retval
 */
void SPI::frequency(int hz)
{
    uint16_t    divider = 0;

    if (hz < 8000) {
        divider = SPI_CLOCK_DIV65536;
    }
    else
    if (hz < 15625) {
        divider = SPI_CLOCK_DIV32768;
    }
    else
    if (hz < 31250) {
        divider = SPI_CLOCK_DIV16384;
    }
    else
    if (hz < 62500) {
        divider = SPI_CLOCK_DIV8192;
    }
    else
    if (hz < 125000) {
        divider = SPI_CLOCK_DIV4096;
    }
    else
    if (hz < 250000) {
        divider = SPI_CLOCK_DIV2048;
    }
    else
    if (hz < 500000) {
        divider = SPI_CLOCK_DIV1024;
    }
    else
    if (hz < 1000000) {
        divider = SPI_CLOCK_DIV512;
    }
    else
    if (hz < 2000000) {
        divider = SPI_CLOCK_DIV256;
    }
    else
    if (hz < 4000000) {
        divider = SPI_CLOCK_DIV128;
    }
    else
    if (hz < 8000000) {
        divider = SPI_CLOCK_DIV64;
    }
    else
    if (hz < 20000000) {
        divider = SPI_CLOCK_DIV32;
    }
    else
    if (hz < 40000000) {
        divider = SPI_CLOCK_DIV16;
    }
    else
    if (hz < 80000000) {
        divider = SPI_CLOCK_DIV8;
    }
    else
    if (hz < 160000000) {
        divider = SPI_CLOCK_DIV4;
    }
    else {

        // hz >= 160000000
        divider = SPI_CLOCK_DIV2;
    }

    setClockDivider(divider);
}

/**
 * @brief
 * @note
 * @param
 * @retval
 */
uint8_t SPI::write(uint8_t value)
{
    volatile uint32_t*  paddr = bcm2835_spi0 + BCM2835_SPI0_CS / 4;
    volatile uint32_t*  fifo  = bcm2835_spi0 + BCM2835_SPI0_FIFO / 4;

    bcm2835_peri_set_bits(paddr, BCM2835_SPI0_CS_CLEAR, BCM2835_SPI0_CS_CLEAR);

    bcm2835_peri_set_bits(paddr, BCM2835_SPI0_CS_TA, BCM2835_SPI0_CS_TA);

    while (!(bcm2835_peri_read(paddr) & BCM2835_SPI0_CS_TXD))
        wait_us(10);

    bcm2835_peri_write_nb(fifo, value);

    while (!(bcm2835_peri_read_nb(paddr) & BCM2835_SPI0_CS_DONE))
        wait_us(10);

    uint32_t    ret = bcm2835_peri_read_nb(fifo);

    bcm2835_peri_set_bits(paddr, 0, BCM2835_SPI0_CS_TA);

    return ret;
}

/**
 * @brief
 * @note
 * @param
 * @retval
 */
int SPI::write(const char* tx_buffer, int tx_length, char* rx_buffer, int rx_length)
{
    int    len = tx_length;
    if (rx_length > len)
        len = rx_length;

    volatile uint32_t*  paddr = bcm2835_spi0 + BCM2835_SPI0_CS / 4;
    volatile uint32_t*  fifo  = bcm2835_spi0 + BCM2835_SPI0_FIFO / 4;

    // This is Polled transfer as per section 10.6.1

    // BUG ALERT: what happens if we get interupted in this section, and someone else
    // accesses a different peripheral?
    // Clear TX and RX fifos
    bcm2835_peri_set_bits(paddr, BCM2835_SPI0_CS_CLEAR, BCM2835_SPI0_CS_CLEAR);

    // Set TA = 1
    bcm2835_peri_set_bits(paddr, BCM2835_SPI0_CS_TA, BCM2835_SPI0_CS_TA);

    int    i;
    for (i = 0; i < len; i++) {

        // Maybe wait for TXD
        while (!(bcm2835_peri_read(paddr) & BCM2835_SPI0_CS_TXD))
            wait_us(10);

        // Write to FIFO, no barrier
        if (i < tx_length)
            bcm2835_peri_write_nb(fifo, tx_buffer[i]);
        else
            bcm2835_peri_write_nb(fifo, _write_fill);

        // Wait for RXD
        while (!(bcm2835_peri_read(paddr) & BCM2835_SPI0_CS_RXD))
            wait_us(10);

        // then read the data byte
        if (i < rx_length)
            rx_buffer[i] = bcm2835_peri_read_nb(fifo);
        else
            bcm2835_peri_read_nb(fifo);
    }

    // Wait for DONE to be set
    while (!(bcm2835_peri_read_nb(paddr) & BCM2835_SPI0_CS_DONE))
        wait_us(10);

    // Set TA = 0, and also set the barrier
    bcm2835_peri_set_bits(paddr, 0, BCM2835_SPI0_CS_TA);

    return len;
}

/**
 * @brief
 * @note
 * @param
 * @retval
 */
void SPI::set_default_write_value(char value)
{
    _write_fill = value;
}

/**
 * @brief
 * @note    The divisor must be a power of 2. Odd numbers rounded down.
 *          The maximum SPI clock rate is of the APB clock.
 * @param   divider Defaults to 0, which means a divider of 65536.
 * @retval
 */
void SPI::setClockDivider(uint16_t divider)
{
    volatile uint32_t*  paddr = bcm2835_spi0 + BCM2835_SPI0_CLK / 4;
    bcm2835_peri_write(paddr, divider);
}


