#ifndef _SPI_H_
#define _SPI_H_

#include "BCM2835.h"

class   SPI
{
public:
    /** Create a SPI master connected to the specified pins.
     */
    SPI();
    ~SPI();

    /** Configure the data transmission format.
     *
     *  @param bits Number of bits per SPI frame (4 - 16).
     *  @param mode Clock polarity and phase mode (0 - 3).
     *
     * @code
     * mode | POL PHA
     * -----+--------
     *   0  |  0   0
     *   1  |  0   1
     *   2  |  1   0
     *   3  |  1   1
     * @endcode
     */
    void    format(int bits, uint8_t mode = 0);

    /** Set the SPI bus clock frequency.
     *
     *  @param hz Clock frequency in Hz (default = 1MHz).
     */
    void    frequency(int hz);

    /** Write to the SPI Slave and return the response.
     *
     *  @param value Data to be sent to the SPI slave.
     *
     *  @return Response from the SPI slave.
     */
    uint8_t write(uint8_t value);

    /** Write to the SPI Slave and obtain the response.
     *
     *  The total number of bytes sent and received will be the maximum of
     *  tx_length and rx_length. The bytes written will be padded with the
     *  value 0xff.
     *
     *  @param tx_buffer Pointer to the byte-array of data to write to the device.
     *  @param tx_length Number of bytes to write, may be zero.
     *  @param rx_buffer Pointer to the byte-array of data to read from the device.
     *  @param rx_length Number of bytes to read, may be zero.
     *  @return
     *      The number of bytes written and read from the device. This is
     *      maximum of tx_length and rx_length.
     */
    int     write(const char* tx_buffer, int tx_length, char* rx_buffer, int rx_length);

    /** Set default write value.
      * SPI requires the master to send some data during a read operation.
      * Different devices may require different default byte values.
      * For example: An SD Card requires default bytes to be 0xFF.
      *
      * @param data Default character to be transmitted during a read operation.
      */
    void    set_default_write_value(char value);
private:
    uint8_t _write_fill;

    void    setClockDivider(uint16_t divider);
};

#endif	// _SPI_H_

