#include "mbed.h"

extern struct bcm2835_peripheral   gpio;

/********** FUNCTIONS OUTSIDE CLASSES **********/

// Write a HIGH or a LOW value to a digital pin
void gpio_write(PinName pin, int value)
{
    if (value == HIGH)
        GPSET0 = (1 << pin);
    else
    if (value == LOW)
        GPCLR0 = (1 << pin);

    wait_us(1); // Delay to allow any change in state to be reflected in the LEVn, register bit.
}

// Reads the value from a specified digital pin, either HIGH or LOW.
int gpio_read(PinName pin)
{
    Digivalue   value;
    if (GPLEV0 & (1 << pin))
        value = HIGH;
    else
        value = LOW;

    return value;
}

// Function select
// pin is a BCM2835 GPIO pin number NOT RPi pin number
//      There are 6 control registers, each control the functions of a block
//      of 10 pins.
//      Each control register has 10 sets of 3 bits per GPIO pin:
//
//      000 = GPIO Pin X is an input
//      001 = GPIO Pin X is an output
//      100 = GPIO Pin X takes alternate function 0
//      101 = GPIO Pin X takes alternate function 1
//      110 = GPIO Pin X takes alternate function 2
//      111 = GPIO Pin X takes alternate function 3
//      011 = GPIO Pin X takes alternate function 4
//      010 = GPIO Pin X takes alternate function 5
//
// So the 3 bits for port X are:

//      X / 10 + ((X % 10) * 3)

void bcm2835_gpio_fsel(uint8_t pin, uint8_t mode)
{
    // Function selects are 10 pins per 32 bit word, 3 bits per pin
    volatile uint32_t*  paddr = (volatile uint32_t*)gpio.map + BCM2835_GPFSEL0 / 4 + (pin / 10);
    uint8_t             shift = (pin % 10) * 3;
    uint32_t            mask = BCM2835_GPIO_FSEL_MASK << shift;
    uint32_t            value = mode << shift;

    bcm2835_peri_set_bits(paddr, value, mask);
}

/**
 * @brief
 * @note
 * @param
 * @retval
 */
void gpio_dir(PinName pin, PinDirection direction)
{
    uint8_t     gpfsel = pin / 10;
    uint8_t     shift = (pin % 10) * 3;
    uint32_t    mask = BCM2835_GPIO_FSEL_MASK << shift;
    uint32_t    outp = BCM2835_GPIO_FSEL_OUTP << shift;

    if (direction == PIN_OUTPUT) {
        *(gpio.addr + gpfsel) &= ~mask;
        *(gpio.addr + gpfsel) |= outp;
    }
    else
    if (direction == PIN_INPUT) {
        *(gpio.addr + gpfsel) &= ~mask;
    }
}

/**
 * @brief
 * @note
 * @param
 * @retval
 */
void gpio_mode(PinName pin, PinMode mode)
{
    mode == PullUp ? gpio_write(pin, HIGH) : gpio_write(pin, LOW);
}

/**
 * @brief
 * @note
 * @param
 * @retval
 */
uint8_t shiftIn(PinName dPin, PinName cPin, bcm2835SPIBitOrder order)
{
    uint8_t value = 0;
    int8_t  i;

    if (order == MSBFIRST)
        for (i = 7; i >= 0; --i) {
            gpio_write(cPin, HIGH);
            value |= gpio_read(dPin) << i;
            gpio_write(cPin, LOW);
        }
    else
        for (i = 0; i < 8; ++i) {
            gpio_write(cPin, HIGH);
            value |= gpio_read(dPin) << i;
            gpio_write(cPin, LOW);
        }

    return value;
}

/**
 * @brief
 * @note
 * @param
 * @retval
 */
void shiftOut(PinName dPin, PinName cPin, bcm2835SPIBitOrder order, uint8_t val)
{
    int8_t  i;

    if (order == MSBFIRST)
        for (i = 7; i >= 0; --i) {
            gpio_write(dPin, val & (1 << i));
            gpio_write(cPin, HIGH);
            gpio_write(cPin, LOW);
        }
    else
        for (i = 0; i < 8; ++i) {
            gpio_write(dPin, val & (1 << i));
            gpio_write(cPin, HIGH);
            gpio_write(cPin, LOW);
        }
}

