//#include "mbed.h"
#include "BCM2835.h"
#include "Thread.h"
#include "ThisThread.h"
#include "Peripheral.h"

struct bcm2835_peripheral   bsc_rev1 = { BCM2835_PERI_BASE + 0X205000, 0, 0, 0 };
struct bcm2835_peripheral   bsc_rev2 = { BCM2835_PERI_BASE + 0X804000, 0, 0, 0 };
struct bcm2835_peripheral   bsc0;
struct bcm2835_peripheral   gpio = { BCM2835_PERI_BASE + 0x200000, 0, 0, 0 };

timeval  start_program, end_point;

off_t                       bcm2835_peripherals_base = BCM2835_PERI_BASE;
size_t                      bcm2835_peripherals_size = BCM2835_PERI_SIZE;
volatile uint32_t*          bcm2835_peripherals;
volatile uint32_t*          bcm2835_pwm  = (uint32_t *)MAP_FAILED;
volatile uint32_t*          bcm2835_clk  = (uint32_t *)MAP_FAILED;
volatile uint32_t*          bcm2835_bsc1 = (uint32_t *)MAP_FAILED;
volatile uint32_t*          bcm2835_spi0 = (uint32_t *)MAP_FAILED;
volatile uint32_t*          bcm2835_st	= (uint32_t *)MAP_FAILED;


//Constructor
Peripheral::Peripheral()
{
    REV = getBoardRev();
    if (map_peripheral(&gpio) == -1) {
        printf("Failed to map the physical GPIO registers into the virtual memory space.\n");
    }

    memfd = -1;

    // Open the master /dev/memory device
    if ((memfd = open("/dev/mem", O_RDWR | O_SYNC)) < 0) {
        fprintf(stderr, "bcm2835_init: Unable to open /dev/mem: %s\n", strerror(errno));
        exit(1);
    }

    bcm2835_peripherals = mapmem("gpio", bcm2835_peripherals_size, memfd, bcm2835_peripherals_base);
    if (bcm2835_peripherals == MAP_FAILED)
        exit(1);

    /* Now compute the base addresses of various peripherals,
    // which are at fixed offsets within the mapped peripherals block
    // Caution: bcm2835_peripherals is uint32_t*, so divide offsets by 4
    */
    bcm2835_pwm  = bcm2835_peripherals + BCM2835_GPIO_PWM/4;
    bcm2835_clk  = bcm2835_peripherals + BCM2835_CLOCK_BASE/4;
    bcm2835_spi0 = bcm2835_peripherals + BCM2835_SPI0_BASE/4;
    bcm2835_bsc1 = bcm2835_peripherals + BCM2835_BSC1_BASE/4; /* I2C */
    bcm2835_st   = bcm2835_peripherals + BCM2835_ST_BASE/4;

    // start timer
    gettimeofday(&start_program, NULL);
}

//Destructor
Peripheral::~Peripheral()
{
    unmap_peripheral(&gpio);

    bcm2835_pwm  = (uint32_t *)MAP_FAILED;
    bcm2835_clk  = (uint32_t *)MAP_FAILED;
    bcm2835_spi0 = (uint32_t *)MAP_FAILED;
    bcm2835_bsc1 = (uint32_t *)MAP_FAILED;
}

/*******************
 * Private methods *
 *******************/

// Exposes the physical address defined in the passed structure using mmap on /dev/mem
int Peripheral::map_peripheral(struct bcm2835_peripheral* p)
{
    // Open /dev/mem

    if ((p->mem_fd = open("/dev/mem", O_RDWR | O_SYNC)) < 0) {
        printf("Failed to open /dev/mem, try checking permissions.\n");
        return -1;
    }

    p->map = mmap
        (
            NULL,
            BLOCK_SIZE,
            PROT_READ | PROT_WRITE,
            MAP_SHARED,
            p->mem_fd,  // File descriptor to physical memory virtual file '/dev/mem'
            p->addr_p   // Address in physical map that we want this memory block to expose
        );

    if (p->map == MAP_FAILED) {
        perror("mmap");
        return -1;
    }

    p->addr = (volatile unsigned int*)p->map;

    return 0;
}

/**
 * @brief
 * @note
 * @param
 * @retval
 */
void Peripheral::unmap_peripheral(struct bcm2835_peripheral* p)
{
    munmap(p->map, BLOCK_SIZE);
    unistd::close(p->mem_fd);
}

//
// Low level convenience functions
//

/**
 * @brief
 * @note
 * @param
 * @retval
 */
uint32_t* mapmem(const char* msg, size_t size, int fd, off_t off)
{
    uint32_t*   map = (uint32_t*)mmap(NULL, size, (PROT_READ | PROT_WRITE), MAP_SHARED, fd, off);
    if (MAP_FAILED == map)
        fprintf(stderr, "bcm2835_init: %s mmap failed: %s\n", msg, strerror(errno));
    return map;
}

// safe read from peripheral
uint32_t bcm2835_peri_read(volatile uint32_t* paddr)
{
    uint32_t    ret = *paddr;
    ret = *paddr;
    return ret;
}

// read from peripheral without the read barrier
uint32_t bcm2835_peri_read_nb(volatile uint32_t* paddr)
{
    return *paddr;
}

// safe write to peripheral
void bcm2835_peri_write(volatile uint32_t* paddr, uint32_t value)
{
    *paddr = value;
    *paddr = value;
}

// write to peripheral without the write barrier
void bcm2835_peri_write_nb(volatile uint32_t* paddr, uint32_t value)
{
    *paddr = value;
}

// Set/clear only the bits in value covered by the mask
void bcm2835_peri_set_bits(volatile uint32_t* paddr, uint32_t value, uint32_t mask)
{
    uint32_t    v = bcm2835_peri_read(paddr);
    v = (v &~mask) | (value & mask);
    bcm2835_peri_write(paddr, v);
}

/**
 * @brief
 * @note
 * @param
 * @retval
 */
int getBoardRev()
{
    FILE*       cpu_info;
    char        line[120];
    char*       c, finalChar;
    //static int  rev = 0;

    if (REV != 0)
        return REV;

    if ((cpu_info = fopen("/proc/cpuinfo", "r")) == NULL) {
        fprintf(stderr, "Unable to open /proc/cpuinfo. Cannot determine board reivision.\n");
        exit(1);
    }

    while (fgets(line, 120, cpu_info) != NULL) {
        if (strncmp(line, "Revision", 8) == 0)
            break;
    }

    fclose(cpu_info);

    if (line == NULL) {
        fprintf(stderr, "Unable to determine board revision from /proc/cpuinfo.\n");
        exit(1);
    }

    for (c = line; *c; ++c)
        if (isdigit(*c))
            break;

    if (!isdigit(*c)) {
        fprintf(stderr, "Unable to determine board revision from /proc/cpuinfo\n");
        fprintf(stderr, "  (Info not found in: %s\n", line);
        exit(1);
    }

    finalChar = c[strlen(c) - 2];

    if ((finalChar == '2') || (finalChar == '3')) {
        bsc0 = bsc_rev1;
        return 1;
    }
    else {
        bsc0 = bsc_rev2;
        return 2;
    }
}

/**
 * @brief
 * @note
 * @param
 * @retval
 */
void attachInterrupt(PinName p, void (*f) (), Digivalue m)
{
    int                 GPIOPin = p;
    pthread_t*          threadId = getThreadIdFromPin(p);
    struct ThreadArg*   threadArgs = (ThreadArg*)malloc(sizeof(ThreadArg));
    threadArgs->func = f;
    threadArgs->pin = GPIOPin;

    //Export pin for interrupt
    FILE*   fp = fopen("/sys/class/gpio/export", "w");
    if (fp == NULL) {
        fprintf(stderr, "Unable to export pin %d for interrupt\n", p);
        exit(1);
    }
    else {
        fprintf(fp, "%d", GPIOPin);
    }

    fclose(fp);

    //The system to create the file /sys/class/gpio/gpio<GPIO number>
    //So we wait a bit
    ThisThread::sleep_for_ms(1);

    char*   interruptFile = NULL;
    asprintf(&interruptFile, "/sys/class/gpio/gpio%d/edge", GPIOPin);

    //Set detection condition
    fp = fopen(interruptFile, "w");
    if (fp == NULL) {
        fprintf(stderr, "Unable to set detection type on pin %d\n", p);
        exit(1);
    }
    else {
        switch (m) {
            case RISING:
                fprintf(fp, "rising");
                break;

            case FALLING:
                fprintf(fp, "falling");
                break;

            default:
                fprintf(fp, "both");
                break;
        }
    }

    fclose(fp);

    if (*threadId == 0) {

        //Create a thread passing the pin and function
        pthread_create(threadId, NULL, threadFunction, (void*)threadArgs);
    }
    else {

        //First cancel the existing thread for that pin
        pthread_cancel(*threadId);

        //Create a thread passing the pin, function and mode
        pthread_create(threadId, NULL, threadFunction, (void*)threadArgs);
    }
}

/**
 * @brief
 * @note
 * @param
 * @retval
 */
void detachInterrupt(PinName p)
{
    int     GPIOPin = p;

    FILE*   fp = fopen("/sys/class/gpio/unexport", "w");
    if (fp == NULL) {
        fprintf(stderr, "Unable to unexport pin %d for interrupt\n", p);
        exit(1);
    }
    else {
        fprintf(fp, "%d", GPIOPin);
    }

    fclose(fp);

    pthread_t*  threadId = getThreadIdFromPin(p);
    pthread_cancel(*threadId);
}

Peripheral  peripheral = Peripheral();


