#include "mbed.h"

#include "Usb.h"
#include "usbhub.h"
#include "pgm_strings.h"

Usb usb(D11, D12, D13, D10, D9);    // mosi, miso, sck, ss, intr

uint8_t getConfDescr(uint8_t addr, uint8_t conf);

void printAllAddresses(UsbDevice* pdev)
{
    UsbDeviceAddress    adr;

    adr.devAddress = pdev->address.devAddress;
    printf("\r\nAddr: 0x%x (0x%x.0x%x.0x%x)\r\n", adr.devAddress, adr.bmHub, adr.bmParent, adr.bmAddress);
}

/**
 * @brief
 * @note
 * @param
 * @retval
 */
void printAddress(uint8_t addr)
{
    UsbDeviceAddress    adr;

    adr.devAddress = addr;
    printf("\r\nADDR:\t0x%x\r\n", adr.devAddress);
    printf("DEV:\t0x%x\r\n", adr.bmAddress);
    printf("PRNT:\t0x%x\r\n", adr.bmParent);
    printf("HUB:\t0x%x\r\n", adr.bmHub);
}

/**
 * @brief
 * @note
 * @param
 * @retval
 */
uint8_t getDevDescr(uint8_t addr, uint8_t& num_conf)
{
    USB_DEVICE_DESCRIPTOR   buf;
    uint8_t                 rcode;

    rcode = usb.getDevDescr(addr, 0, 0x12, (uint8_t*) &buf);
    if (rcode) {
        return(rcode);
    }

    printf("%s", Dev_Header_str);
    printf("%s0x%.8x", Dev_Length_str, buf.bLength);
    printf("%s0x%.8x", Dev_Type_str, buf.bDescriptorType);
    printf("%s0x%.16x", Dev_Version_str, buf.bcdUSB);
    printf("%s0x%.8x", Dev_Class_str, buf.bDeviceClass);
    printf("%s0x%.8x", Dev_Subclass_str, buf.bDeviceSubClass);
    printf("%s0x%.8x", Dev_Protocol_str, buf.bDeviceProtocol);
    printf("%s0x%.8x", Dev_Pktsize_str, buf.bMaxPacketSize0);
    printf("%s0x%.16x", Dev_Vendor_str, buf.idVendor);
    printf("%s0x%.16x", Dev_Product_str, buf.idProduct);
    printf("%s0x%.16x", Dev_Revision_str, buf.bcdDevice);
    printf("%s0x%.8x", Dev_Mfg_str, buf.iManufacturer);
    printf("%s0x%.8x", Dev_Prod_str, buf.iProduct);
    printf("%s0x%.8x", Dev_Serial_str, buf.iSerialNumber);
    printf("%s0x%.8x", Dev_Nconf_str, buf.bNumConfigurations);
    num_conf = buf.bNumConfigurations;
    return(0);
}

/* function to print configuration descriptor */
void printConfDescr(uint8_t* descr_ptr)
{
    USB_CONFIGURATION_DESCRIPTOR*   conf_ptr = (USB_CONFIGURATION_DESCRIPTOR*)descr_ptr;

    printf("%s", Conf_Header_str);
    printf("%s0x%.16x", Conf_Totlen_str, conf_ptr->wTotalLength);
    printf("%s0x%.8x", Conf_Nint_str, conf_ptr->bNumInterfaces);
    printf("%s0x%.8x", Conf_Value_str, conf_ptr->bConfigurationValue);
    printf("%s0x%.8x", Conf_String_str, conf_ptr->iConfiguration);
    printf("%s0x%.8x", Conf_Attr_str, conf_ptr->bmAttributes);
    printf("%s0x%.8x", Conf_Pwr_str, conf_ptr->bMaxPower);
    return;
}

/* function to print interface descriptor */
void printIntfDescr(uint8_t* descr_ptr)
{
    USB_INTERFACE_DESCRIPTOR*   intf_ptr = (USB_INTERFACE_DESCRIPTOR*)descr_ptr;

    printf("%s", Int_Header_str);
    printf("%s0x%.8x", Int_Number_str, intf_ptr->bInterfaceNumber);
    printf("%s0x%.8x", Int_Alt_str, intf_ptr->bAlternateSetting);
    printf("%s0x%.8x", Int_Endpoints_str, intf_ptr->bNumEndpoints);
    printf("%s0x%.8x", Int_Class_str, intf_ptr->bInterfaceClass);
    printf("%s0x%.8x", Int_Subclass_str, intf_ptr->bInterfaceSubClass);
    printf("%s0x%.8x", Int_Protocol_str, intf_ptr->bInterfaceProtocol);
    printf("%s0x%.8x", Int_String_str, intf_ptr->iInterface);
    return;
}

/* function to print endpoint descriptor */
void printEpDescr(uint8_t* descr_ptr)
{
    USB_ENDPOINT_DESCRIPTOR*    ep_ptr = (USB_ENDPOINT_DESCRIPTOR*)descr_ptr;

    printf("%s", End_Header_str);
    printf("%s0x%.8x", End_Address_str, ep_ptr->bEndpointAddress);
    printf("%s0x%.8x", End_Attr_str, ep_ptr->bmAttributes);
    printf("%s0x%.16x", End_Pktsize_str, ep_ptr->wMaxPacketSize);
    printf("%s0x%.8x", End_Interval_str, ep_ptr->bInterval);

    return;
}

/**
 * @brief
 * @note
 * @param
 * @retval
 */
void printHubDescr(uint8_t* descrptr, uint8_t addr)
{
    HubDescriptor*  pHub = (HubDescriptor*)descrptr;
    uint8_t         len = *((uint8_t*)descrptr);

    printf("\r\n\r\nHub Descriptor:\r\n");
    printf("bDescLength:\t\t0x%x\r\n", pHub->bDescLength);
    printf("bDescriptorType:\t0x%x\r\n", pHub->bDescriptorType);
    printf("bNbrPorts:\t\t0x%x\r\n", pHub->bNbrPorts);
    printf("LogPwrSwitchMode:\t0x%x\r\n", pHub->LogPwrSwitchMode);
    printf("CompoundDevice:\t\t0x%x\r\n", pHub->CompoundDevice);
    printf("OverCurrentProtectMode:\t0x%x", pHub->OverCurrentProtectMode);
    printf("TTThinkTime:\t\t0x%x", pHub->TTThinkTime);
    printf("PortIndicatorsSupported:0x%x", pHub->PortIndicatorsSupported);
    printf("Reserved:\t\t0x%x", pHub->Reserved);
    printf("bPwrOn2PwrGood:\t\t0x%x", pHub->bPwrOn2PwrGood);
    printf("bHubContrCurrent:\t0x%x", pHub->bHubContrCurrent);

    for (uint8_t i = 7; i < len; i++)
        printf("0x%.8x ", descrptr[i]);

    printf("\r\n");

    //for (uint8_t i=1; i<=pHub->bNbrPorts; i++)
    //    PrintHubPortStatus(&Usb, addr, i, 1);
}

/**
 * @brief
 * @note
 * @param
 * @retval
 */
void printDescriptors(uint8_t addr)
{
    uint8_t rcode = 0;
    uint8_t num_conf = 0;

    rcode = getDevDescr((uint8_t) addr, num_conf);
    if (rcode) {
        printf("%s0x%.8x", Gen_Error_str, rcode);
    }

    printf("\r\n");

    for (int i = 0; i < num_conf; i++) {
        rcode = getConfDescr(addr, i);  // get configuration descriptor
        if (rcode) {
            printf("%s0x%.8x", Gen_Error_str, rcode);
        }

        printf("\r\n");
    }
}

/**
 * @brief
 * @note
 * @param
 * @retval
 */
void printAllDescriptors(UsbDevice* pdev)
{
    printf("\r\n");
    printf("0x%.8x\r\n--", pdev->address.devAddress);
    printDescriptors(pdev->address.devAddress);
}

/*function to print unknown descriptor */
void printUnknownDescr(uint8_t* descr_ptr)
{
    uint8_t length = *descr_ptr;
    uint8_t i;

    printf("%s", Unk_Header_str);
    printf("%s0x%.8x", Unk_Length_str, *descr_ptr);
    printf("%s0x%.8x", Unk_Type_str, *(descr_ptr + 1 ));
    printf("%s", Unk_Contents_str);
    descr_ptr += 2;
    for (i = 0; i < length; i++) {
        printf("0x%.8x", *descr_ptr);
        descr_ptr++;
    }
}

/**
 * @brief
 * @note
 * @param
 * @retval
 */
uint8_t getConfDescr(uint8_t addr, uint8_t conf)
{
    uint8_t     buf[BUFSIZE];
    uint8_t*    buf_ptr = buf;
    uint8_t     rcode;
    uint8_t     descr_length;
    uint8_t     descr_type;
    uint16_t    total_length;

    rcode = usb.getConfDescr(addr, 0, 4, conf, buf);            //get total length
    LOBYTE(total_length) = buf[2];
    HIBYTE(total_length) = buf[3];
    if (total_length > 256) {

        //check if total length is larger than buffer
        printf("%s", Conf_Trunc_str);
        total_length = 256;
    }

    rcode = usb.getConfDescr(addr, 0, total_length, conf, buf); //get the whole descriptor
    while (buf_ptr < buf + total_length) {

        //parsing descriptors
        descr_length = *(buf_ptr);
        descr_type = *(buf_ptr + 1);
        switch (descr_type) {
            case (USB_DESCRIPTOR_CONFIGURATION):
                printConfDescr(buf_ptr);
                break;

            case (USB_DESCRIPTOR_INTERFACE):
                printIntfDescr(buf_ptr);
                break;

            case (USB_DESCRIPTOR_ENDPOINT):
                printEpDescr(buf_ptr);
                break;

            case 0x29:
                printHubDescr(buf_ptr, addr);
                break;

            default:
                printUnknownDescr(buf_ptr);
                break;
        }                                   //switch( descr_type

        buf_ptr = (buf_ptr + descr_length); //advance buffer pointer
    }                                       //while( buf_ptr <=...

    return(rcode);
}


/**
 * @brief
 * @note
 * @param
 * @retval
 */
int main()
{
    printf("Starting..\r\n");

    if (usb.init() == -1)
        printf("OSC did not start.\r\n");

    wait_ms(200);

    while (true) {
        usb.task();

        if (usb.getUsbTaskState() == USB_STATE_RUNNING) {
            usb.ForEachUsbDevice(&printAllDescriptors);
            usb.ForEachUsbDevice(&printAllAddresses);

            while (1) {}
        }
    }
}
