#include "MAX3421E.h"

uint8_t     MAX3421E::vbusState = 0;

/* constructor */
MAX3421E::MAX3421E(PinName mosi, PinName miso, PinName sck, PinName ss, PinName intr) :
    _spi(mosi, miso, sck),
    _ss(ss),
    _intr(intr)
{}

/* write single byte into MAX3421 register */
void MAX3421E::regWr(uint8_t reg, uint8_t data)
{
    _ss = 0;
    char    tx[2];
    tx[0] = reg | 0x02;
    tx[1] = data;
    _spi.write(tx, 2, NULL, 2);
    _ss = 1;
    return;
}

/* multiple-byte write                            */

/* returns a pointer to memory position after last written */
uint8_t* MAX3421E::bytesWr(uint8_t reg, uint8_t nbytes, uint8_t* data_p)
{
    _ss = 0;
    const char data = reg | 0x02;
    _spi.write(&data, 1, NULL, 1);
    _spi.write((const char*)data_p, nbytes, NULL, nbytes);
    data_p += nbytes;
    _ss = 1;
    return(data_p);
}

/* GPIO write                                           */
/*GPIO byte is split between 2 registers, so two writes are needed to write one byte */

/* GPOUT bits are in the low nibble. 0-3 in IOPINS1, 4-7 in IOPINS2 */
void MAX3421E::gpioWr(uint8_t data)
{
    regWr(rIOPINS1, data);
    data >>= 4;
    regWr(rIOPINS2, data);
    return;
}

/* single host register read    */
uint8_t MAX3421E::regRd(uint8_t reg)
{
    _ss = 0;
    _spi.write((const char*)&reg, 1, NULL, 1);
    uint8_t rv = 0;
    _spi.write((const char*)&rv, 1, (char*)&rv, 1);
    _ss = 1;
    return(rv);
}

/* multiple-byte register read  */

/* returns a pointer to a memory position after last read   */
uint8_t* MAX3421E::bytesRd(uint8_t reg, uint8_t nbytes, uint8_t* data_p)
{
    _ss = 0;
    _spi.write((const char*)&reg, 1, NULL, 1);
    memset(data_p, 0, nbytes);  // Make sure we send out empty bytes
    _spi.write((const char*)data_p, nbytes, (char*)data_p, nbytes);
    data_p += nbytes;
    _ss = 1;
    return(data_p);
}

/* GPIO read. See gpioWr for explanation */
/** @brief  Reads the current GPI input values
*   @retval uint8_t Bitwise value of all 8 GPI inputs
*/

/* GPIN pins are in high nibbles of IOPINS1, IOPINS2    */
uint8_t MAX3421E::gpioRd()
{
    uint8_t gpin = 0;
    gpin = regRd(rIOPINS2);         //pins 4-7
    gpin &= 0xf0;                   //clean lower nibble
    gpin |= (regRd(rIOPINS1) >> 4); //shift low bits and OR with upper from previous operation.
    return(gpin);
}

/** @brief  Reads the current GPI output values
*   @retval uint8_t Bitwise value of all 8 GPI outputs
*/

/* GPOUT pins are in low nibbles of IOPINS1, IOPINS2    */
uint8_t MAX3421E::gpioRdOutput()
{
    uint8_t gpout = 0;
    gpout = regRd(rIOPINS1);            //pins 0-3
    gpout &= 0x0f;                      //clean upper nibble
    gpout |= (regRd(rIOPINS2) << 4);    //shift high bits and OR with lower from previous operation.
    return(gpout);
}

/* reset MAX3421E. Returns number of cycles it took for PLL to stabilize after reset
  or zero if PLL haven't stabilized in 65535 cycles */
uint16_t MAX3421E::reset()
{
    uint16_t    i = 0;
    regWr(rUSBCTL, bmCHIPRES);
    regWr(rUSBCTL, 0x00);
    while (++i) {
        if ((regRd(rUSBIRQ) & bmOSCOKIRQ)) {
            break;
        }
    }

    return(i);
}

/* initialize MAX3421E. Set Host mode, pullups, and stuff. Returns 0 if success, -1 if not */
int8_t MAX3421E::init()
{
    // Moved here.

    // you really should not init hardware in the constructor when it involves locks.
    // Also avoids the vbus flicker issue confusing some devices.
    /* pin and peripheral setup */
    _ss = 1;
    _spi.frequency(26000000);

    /* MAX3421E - full-duplex SPI, level interrupt */
    // GPX pin on. Moved here, otherwise we flicker the vbus.
    regWr(rPINCTL, (bmFDUPSPI | bmINTLEVEL));

    if (reset() == 0) {

        //OSCOKIRQ hasn't asserted in time
        return(-1);
    }

    regWr(rMODE, bmDPPULLDN | bmDMPULLDN | bmHOST); // set pull-downs, Host
    regWr(rHIEN, bmCONDETIE | bmFRAMEIE);           //connection detection

    /* check if device is connected */
    regWr(rHCTL, bmSAMPLEBUS);                      // sample USB bus
    while (!(regRd(rHCTL) & bmSAMPLEBUS));

    //wait for sample operation to finish
    busprobe();                                     //check if anything is connected
    regWr(rHIRQ, bmCONDETIRQ);                      //clear connection detect interrupt
    regWr(rCPUCTL, 0x01);                           //enable interrupt pin
    return(0);
}

/* initialize MAX3421E. Set Host mode, pullups, and stuff. Returns 0 if success, -1 if not */
int8_t MAX3421E::init(int mseconds)
{
    // Moved here.

    // you really should not init hardware in the constructor when it involves locks.
    // Also avoids the vbus flicker issue confusing some devices.
    /* pin and peripheral setup */
    _ss = 1;
    _spi.frequency(26000000); // MAX3421E max frequency

    /* MAX3421E - full-duplex SPI, level interrupt, vbus off */
    regWr(rPINCTL, (bmFDUPSPI | bmINTLEVEL | GPX_VBDET));

    if (reset() == 0) {

        //OSCOKIRQ hasn't asserted in time
        return(-1);
    }

    // Delay a minimum of 1 second to ensure any capacitors are drained.
    // 1 second is required to make sure we do not smoke a Microdrive!
    if (mseconds < 1000)
        mseconds = 1000;
    wait_ms(mseconds);

    regWr(rMODE, bmDPPULLDN | bmDMPULLDN | bmHOST); // set pull-downs, Host
    regWr(rHIEN, bmCONDETIE | bmFRAMEIE);           //connection detection

    /* check if device is connected */
    regWr(rHCTL, bmSAMPLEBUS);                      // sample USB bus
    while (!(regRd(rHCTL) & bmSAMPLEBUS));

    //wait for sample operation to finish
    busprobe();                                     //check if anything is connected
    regWr(rHIRQ, bmCONDETIRQ);                      //clear connection detect interrupt
    regWr(rCPUCTL, 0x01);                           //enable interrupt pin

    // GPX pin on. This is done here so that busprobe will fail if we have a switch connected.
    regWr(rPINCTL, (bmFDUPSPI | bmINTLEVEL));

    return(0);
}

/* probe bus to determine device presence and speed and switch host to this speed */
void MAX3421E::busprobe()
{
    uint8_t bus_sample;
    bus_sample = regRd(rHRSL);              //Get J,K status
    bus_sample &= (bmJSTATUS | bmKSTATUS);  //zero the rest of the byte
    switch (bus_sample) {                   //start full-speed or low-speed host
        case (bmJSTATUS):
            if ((regRd(rMODE) & bmLOWSPEED) == 0) {
                regWr(rMODE, MODE_FS_HOST); //start full-speed host
                vbusState = FSHOST;
            }
            else {
                regWr(rMODE, MODE_LS_HOST); //start low-speed host
                vbusState = LSHOST;
            }
            break;

        case (bmKSTATUS):
            if ((regRd(rMODE) & bmLOWSPEED) == 0) {
                regWr(rMODE, MODE_LS_HOST); //start low-speed host
                vbusState = LSHOST;
            }
            else {
                regWr(rMODE, MODE_FS_HOST); //start full-speed host
                vbusState = FSHOST;
            }
            break;

        case (bmSE1):                       //illegal state
            vbusState = SE1;
            break;

        case (bmSE0):                       //disconnected state
            regWr(rMODE, bmDPPULLDN | bmDMPULLDN | bmHOST | bmSEPIRQ);
            vbusState = SE0;
            break;
    }                                       //end switch( bus_sample )
}

/* MAX3421 state change task and interrupt handler */
uint8_t MAX3421E::task(void)
{
    uint8_t rcode = 0;
    uint8_t pinvalue;
    //USB_HOST_SERIAL.print("Vbus state: ");

    //USB_HOST_SERIAL.println( vbusState, HEX );
    pinvalue = _intr;   //Read();

    //pinvalue = digitalRead( MAX_INT );
    if (pinvalue == 0) {
        rcode = intHandler();
    }

    //    pinvalue = digitalRead( MAX_GPX );
    //    if( pinvalue == LOW ) {
    //        GpxHandler();
    //    }
    //    usbSM();                                //USB state machine
    return(rcode);
}

/**
 * @brief
 * @note
 * @param
 * @retval
 */
uint8_t MAX3421E::intHandler()
{
    uint8_t HIRQ;
    uint8_t HIRQ_sendback = 0x00;
    HIRQ = regRd(rHIRQ);    //determine interrupt source

    //if( HIRQ & bmFRAMEIRQ ) {               //->1ms SOF interrupt handler
    //    HIRQ_sendback |= bmFRAMEIRQ;
    //}//end FRAMEIRQ handling
    if (HIRQ & bmCONDETIRQ) {
        busprobe();
        HIRQ_sendback |= bmCONDETIRQ;
    }

    /* End HIRQ interrupts handling, clear serviced IRQs    */
    regWr(rHIRQ, HIRQ_sendback);
    return(HIRQ_sendback);
}

//template< typename SPI_SS, typename INTR >
//uint8_t MAX3421E::GpxHandler()
//{
//    uint8_t GPINIRQ = regRd( rGPINIRQ );          //read GPIN IRQ register
////    if( GPINIRQ & bmGPINIRQ7 ) {            //vbus overload
////        vbusPwr( OFF );                     //attempt powercycle
////        wait_ms( 1000 );
////        vbusPwr( ON );
////        regWr( rGPINIRQ, bmGPINIRQ7 );
////    }
//    return( GPINIRQ );
//}
