/*
 * example program for the TFTPServer library
 *
 * Copyright (c) 2011 Jaap Vermaas
 * Modified by Zoltan Hudak 2018 for MBED-OS5
 *
 *   For more detaisl see https://os.mbed.com/users/hudakz/code/TFTPServerTest/
 *
 *   The TFTPServer is part of the LaOS project (see: http://wiki.laoslaser.org)
 *
 *   LaOS is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   LaOS is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with LaOS.  If not, see <http://www.gnu.org/licenses/>.
 * 
 */
#include "mbed.h"
#include "SDBlockDevice.h"
#include "FATFileSystem.h"
#include "EthernetInterface.h"
#include "TFTPServer.h"
#include "errno.h"

Serial              pc(USBTX, USBRX);
SDBlockDevice       bd(p5, p6, p7, p8); // SD card SPI driver pins: mosi, miso, sclk, cs
FATFileSystem       fs("fs");           // FAT file sytem
EthernetInterface   net;                // Ethernet interface
TFTPServer*         server;             // Pointer to a TFTPServer
int                 myPort = 69;        // TFPTServer port

#define USE_DHCP

/**
 * @brief
 * @note
 * @param
 * @retval
 */
int main()
{
    // Try to mount the filesystem. This should work without the Ethernet.
    pc.printf("Mounting the filesystem... ");

    int err = fs.mount(&bd);
    pc.printf("%s\r\n", (err ? "Failed :(\r\n" : "OK\r\n"));
    if (err)
        return err;

    // Open a file. This should work without the Ethernet, too.
    pc.printf("Opening file '/fs/test.txt'... ");

    FILE*   fp = fopen("/fs/test.txt", "w+");
    pc.printf("%s\r\n", (!fp ? "Failed :(\r\n" : "OK\r\n"));

    if (!fp)
    {
        // Create the test file if it doesn't exist
        pc.printf("No file found, creating a new file ...\r\n");
        fp = fopen("/fs/test.txt", "w+");
        pc.printf("%s\r\n", (!fp ? "Failed :(" : "OK"));
        if (!fp)
        {
            error("error: %s (%d)\r\n", strerror(errno), -errno);
            return errno;
        }
    }
    
    for (int i = 0; i < 10; i++)
    {
        pc.printf("Writing numbers (%d/%d)... ", i, 10);
        err = fprintf(fp, "    %d\r\n", i);
        if (err < 0)
        {
            pc.printf("Fail :(\r\n");
            error("error: %s (%d)\r\n", strerror(errno), -errno);
        }
        else
            pc.printf("OK\r\n");
    }

    pc.printf("Writing numbers (%d/%d)... OK\r\n\r\n", 10, 10);
    
    err = fclose(fp);
    pc.printf("Closing file '/fs/test.txt'... ");
    pc.printf("%s\r\n", (err ? "Failed :(\r\n" : "OK\r\n"));
    if (err)
      return err;

    // Connect to the Ethernet
    pc.printf("Connecting to the Ethernet using ");
#ifdef USE_DHCP
    pc.printf("DHCP server... ");
#else
    pc.printf("fixed IP ...\r\n");
    net.set_network("192.168.1.185", "255.255.255.0", "192.168.1.1");
#endif
    err = net.connect();
    pc.printf("%s\r\n", (err ? "Failed :(\r\n" : "OK\r\n"));
    if (err)
        return err;

    pc.printf("IP:      %s\r\n", net.get_ip_address());
    pc.printf("NetMask: %s\r\n\r\n", net.get_netmask());

    // Create a TFTP server.
    pc.printf("Creating a TFTP server... ");
    server = new TFTPServer(&net, myPort);
    pc.printf("%s\r\n", (server ? "OK" : "Failed :("));
    if (!server)
        return 1;
    
    if (server->getState() == TFTPServer::LISTENING)
        pc.printf("\r\nThe TFTP server is listening at port %d.\r\n", myPort);
    else
    {
        pc.printf("\r\nSomething went wrong.\r\nThe TFTP server is not listening.\r\n");
        return 1;
    }

    int     fileCount = 0;  // incoming files
    char    fileName[256];  // to display filenames

    while (1)
    {
        server->poll();

        if (server->fileCount() > fileCount)
        {
            fileCount = server->fileCount();
            server->getFileName(fileName);
            pc.printf("new file: %s\r\n", fileName);
        }
    }
}
