/*
 * TCP/IP to Serial Converter.
 *
 *  Created: 2015-04-10
 *   Author: Zoltan Hudak
 *
 *  ENC28J60 modul is connected to the mbed board to create a TCP/IP server.
 *  The server is listening for clients at IP address 192.168.1.181, port 10001 (you can adapt it to your needs).
 *  Binary data received from a client is transmitted over serial device and
 *  in turn data received from the serial device is sent back to the TCP/IP client.
 *
 *  How to use the converter:
 *  Compile and save it to your mbed board.
 *  Connect a ENC28J60 module to the mber board according to the pin assignment below.
 *  Connect a serial device to the mbed serial port. Please notice that in case your serial device
 *  is using an RS-232 connection a "serial TTL to serial RS-232" converter is needed!
 *  To test or use this "TCP/IP to Serial Converter", create a virtual serial port on your PC as follows:
 *  See "Using HW VSP powered by HW group" at <http://www.HW-group.com>
 *  and download their free "HW VPS3 Virtual Serial Port" program from <http://www.hw-group.com/products/hw_vsp/index_en.html>
 *  Install and run the application and go to tab "Virtual Serial Port".
 *  Create a new virtual serial port, for example with name COM200, IP Address 192.168.1.181 and Port number 10001.
 *  From now on, whatever data you send to COM200 on your PC will be redirected via Ethernet to this "TCP/IP to serial Converter"
 *  which will pass it to the connected remote serial device. 
 *  In turn, the response from the remote serial device will be sent back to COM200 on your PC.
 *
 */

#include "mbed.h"
#include "UIPEthernet.h"

const int           BAUD = 115200;  // serial bit rate (change to match the speed of the connected serial device)
uint8_t             rxBuf[512];     // serial Rx buffer (adjust the size to your needs)
uint8_t             txBuf[512];     // serial Tx buffer (adjust the size to your needs)
// MAC number must be unique within the connected network. Modify as appropriate.
const uint8_t       MY_MAC[6] = { 0x00, 0x01, 0x02, 0x03, 0x04, 0x06 };
// IP address must be also unique and compatible with your network. Change as appropriate.
const IPAddress     MY_IP(192, 168, 1, 181);
// TCP/IP port (select a port)
const uint16_t      MY_PORT = 10001;


// UIPEthernet is the name of a global instance of UIPEthernetClass.
// Do not change the name! It is used within the UIPEthernet library.
#if defined(TARGET_LPC1768)
Serial              serial(p9, p10);                        // serial: tx, rx
UIPEthernetClass    UIPEthernet(p11, p12, p13, p8);         // spi: mosi, miso, sck, cs
#elif defined(TARGET_NUCLEO_F103RB) || defined(TARGET_NUCLEO_L152RE) || defined(TARGET_NUCLEO_F030R8)  \
   || defined(TARGET_NUCLEO_F401RE) || defined(TARGET_NUCLEO_F302R8) || defined(TARGET_NUCLEO_L053R8)  \
   || defined(TARGET_NUCLEO_F411RE) || defined(TARGET_NUCLEO_F334R8) || defined(TARGET_NUCLEO_F072RB)  \
   || defined(TARGET_NUCLEO_F091RC) || defined(TARGET_NUCLEO_F303RE) || defined(TARGET_NUCLEO_F070RB)  \
   || defined(TARGET_KL25Z ) || defined(TARGET_KL46Z) || defined(TARGET_K64F) || defined(TARGET_KL05Z) \
   || defined(TARGET_K20D50M) || defined(TARGET_K22F) \
   || defined(TARGET_NRF51822) \
   || defined(TARGET_RZ_A1H)
Serial              serial(PA_2, PA_3);                     // serial: tx, rx
UIPEthernetClass    UIPEthernet(D11, D12, D13, D10);        // spi: mosi, miso, sck, cs
#else
// If your board/plaform is not present yet then uncomment
// the following three lines and replace TARGET_YOUR_BOARD and other pins as appropriate.

//#elif defined(TARGET_YOUR_BOARD)
//Serial              serial(SERIAL_TX, SERIAL_RX);                         // serial: tx, rx
//UIPEthernetClass    UIPEthernet(SPI_MOSI, SPI_MISO, SPI_SCK, SPI_CS);     // spi: mosi, miso, sck, cs
#endif

EthernetServer      myServer = EthernetServer(MY_PORT); // create server
volatile int        rxLength = 0;                       // number of serial bytes received

/**
 * @brief   Reads data from serial port
 * @note    Called on arrival of new serial data
 * @param
 * @retval
 */
void onSerial(void) {
    while(serial.readable())
        rxBuf[rxLength++] = serial.getc();   // read serial data
}

/**
 * @brief   Main
 * @note
 * @param
 * @retval
 */
int main(void) {
    serial.attach(&onSerial);

    while(1) {
        EthernetClient  client = myServer.available();
        if(client) {                                        // check for client connected
            size_t  txLength = client.available();
            if(txLength > 0) {
                rxLength = 0;                               // clear rx length
                txLength = client.read(txBuf, txLength);    // read data received from the TCP/IP client
                for(int i = 0; i < txLength; i++)           // send through serial
                    serial.putc(txBuf[i]);
                wait_ms(50);                                // receive serial data from device
                client.write(rxBuf, rxLength);              // send data to client over ethernet
            }
        }
    }
}
